CREATE OR REPLACE FUNCTION ci.set_anyelement( _name varchar, _value anyelement ) RETURNS anyelement AS $$
  BEGIN

      IF _value IS NULL THEN
          RAISE EXCEPTION 'value for variable »%« is null, use ci.delete( _name ); to unset ci vars', _name;
      END IF;
      
      INSERT INTO ci.vars( name, type, content ) 
           VALUES ( _name, pg_typeof( _value )::text, _value )
      ON CONFLICT ( name ) 
      DO UPDATE SET content = _value, type = pg_typeof( _value )::text;
      
      RETURN _value;
      
  END $$ LANGUAGE plpgsql VOLATILE;


  
CREATE OR REPLACE FUNCTION ci.get_anyelement(
      _name     text,
      _default  anyelement
  ) RETURNS anyelement AS $$
  DECLARE
      _data         text;
      _stored_type  text;
      _current_type text = pg_typeof( _default )::text;
  BEGIN
      
      _data := ci.get( _name, _default::text );
      
      -- type validation
      _stored_type  := ci.get_type( _name );
      
      IF _stored_type <> _current_type THEN
          RAISE EXCEPTION 'ci variable »%« is of type »%«, not »%«', _name, _stored_type, _current_type;
      END IF;
      
      RETURN _data;
      
  END $$ LANGUAGE plpgsql VOLATILE;
  

CREATE OR REPLACE FUNCTION ci.delete( _name varchar ) RETURNS void AS $$
      DELETE FROM ci.vars WHERE name = _name;
  $$ LANGUAGE sql VOLATILE;

CREATE OR REPLACE FUNCTION ci.get_type( _name varchar ) RETURNS text AS $$
      SELECT type FROM ci.vars WHERE name = _name;
  $$ LANGUAGE sql VOLATILE;

CREATE OR REPLACE FUNCTION ci.get(
      _name     text,       -- ci variable
      _default  text = null -- default value if ci variable is not set
  ) RETURNS text AS $$
  DECLARE
      _data text = null;
      _unset_exception CONSTANT text := 'ci variable »%s« is not set';
  BEGIN
      
      _data := content FROM ci.vars WHERE name = _name;
      
      IF _data IS NULL THEN
      
          IF _default IS NOT NULL THEN
              RETURN _default;
          END IF;
          
          RAISE EXCEPTION '%', format( _unset_exception, _name );
      
      END IF;
      
      RETURN _data;
      
  END $$ LANGUAGE plpgsql VOLATILE;


-- this will generated all of those functions
-- 
-- ci.set( _name, _value::bool ); 
-- ci.bool( _name );
-- ci.set( _name, _value::char ); 
-- ci.char( _name );
-- ci.set( _name, _value::text ); 
-- ci.text( _name );
-- ci.set( _name, _value::varchar ); 
-- ci.varchar( _name );
-- ci.set( _name, _value::numeric ); 
-- ci.numeric( _name );
-- ci.set( _name, _value::float ); 
-- ci.float( _name );
-- ci.set( _name, _value::int ); 
-- ci.int( _name );
-- ci.set( _name, _value::bigint ); 
-- ci.bigint( _name );
-- ci.set( _name, _value::smallint ); 
-- ci.smallint( _name );
-- ci.set( _name, _value::time ); 
-- ci.time( _name );
-- ci.set( _name, _value::date ); 
-- ci.date( _name );
-- ci.set( _name, _value::interval ); 
-- ci.interval( _name );
-- ci.set( _name, _value::timestamp ); 
-- ci.timestamp( _name );
-- ci.set( _name, _value::timestamptz ); 
-- ci.timestamptz( _name );
-- ci.set( _name, _value::tsrange ); 
-- ci.tsrange( _name );
-- ci.set( _name, _value::tstzrange ); 
-- ci.tstzrange( _name );
-- ci.set( _name, _value::daterange ); 
-- ci.daterange( _name );
-- ci.set( _name, _value::json ); 
-- ci.json( _name );
-- ci.set( _name, _value::jsonb ); 
-- ci.jsonb( _name );
-- ci.set( _name, _value::bytea ); 
-- ci.bytea( _name );
-- ci.set( _name, _value::regclass ); 
-- ci.regclass( _name );
-- ci.set( _name, _value::oid ); 
-- ci.oid( _name );
-- 
-- aswell as the ci.<type>_array() versions
DO $$
DECLARE
    _type text;
    _supported_types text[] = array[
    
        -- bool :D
        'bool',
        
        -- strings
        'char', 'text', 'varchar',
        
        -- numbers
        'numeric', 'float',
        'int', 'bigint', 'smallint',        
        
        -- date and time
        'time', 'date', 'interval',
        'timestamp', 'timestamptz',
        'tsrange', 'tstzrange',
        'daterange',
        
        -- nosql
        'json', 'jsonb',
        
        -- stuff
        'bytea', 'regclass', 'oid'
    ];
    
    -- will for example result in ci.set( _name::varchar, _value::int );
    _setter text = 
        $func$
            
            CREATE OR REPLACE FUNCTION ci.set( _name varchar, _value %s ) 
            RETURNS %s AS
            $setter$
            
                SELECT ci.set_anyelement( _name, _value )
                
            $setter$ LANGUAGE sql VOLATILE;
            
        $func$
    ;
    
    -- will for example result in ci.int( _name::varchar, _default::int = null );
    _getter text = 
        $func$
            CREATE OR REPLACE FUNCTION ci.%s( _name varchar, _default %s = null ) 
            RETURNS %s AS 
            $getter$
            
                SELECT ci.get_anyelement( _name, _default )::%s
                
            $getter$ LANGUAGE sql VOLATILE;
        $func$
    ;
    
BEGIN
  
    FOREACH _type IN ARRAY _supported_types LOOP
    
        -- simple datatypes
        EXECUTE format( _setter, _type, _type );
        EXECUTE format( _getter, _type, _type, _type , _type );
        
        -- datatypes as arrays
        _type := _type || '[]';
        
        EXECUTE format( _setter, _type, _type );
        EXECUTE format( _getter, replace( _type, '[]', '_array'), _type, _type , _type );
        
    END LOOP;
  
END $$;
