CREATE OR REPLACE FUNCTION x_950_import.chk__me__translate(
    _me varchar DEFAULT 'st'
  )
  RETURNS varchar
  LANGUAGE plpgsql
  IMMUTABLE
  AS $$
  BEGIN
    CASE lower(_me)
      WHEN 'st' THEN RETURN 'Stk';
      WHEN 'kg' THEN RETURN 'kg';
      WHEN 'm'  THEN RETURN 'm';
      WHEN 'rl' THEN RETURN 'RO';
      WHEN ''  THEN RETURN 'Stk';   --- ist leer
      ELSE
        RAISE EXCEPTION 'FUNCTION x_950_import.chk__me__translate: ME ist nicht erkannt - %', _me;
    END CASE;
  END;
  $$;

CREATE OR REPLACE FUNCTION api.chk_stv_import_json__from_x_950_import(
    p_filename text
  )
  RETURNS json
  LANGUAGE plpgsql
  AS $$
  DECLARE
    c_me              integer := 0;
    c_art_haupt       integer := 0;
    c_art_pos         integer := 0;
    c_art             integer := 0;
    c_artmgc_std      integer := 0;
    c_artmgc_pos      integer := 0;
    c_artmgc          integer := 0;
    c_stv             integer := 0;
    c_del             integer := 0;
    v_rows            integer := 0;
  BEGIN
    --------------------------------------------------------------------
    -- Stammdaten/ME „RO“ (einmalig) anlegen
    --------------------------------------------------------------------
    INSERT INTO mgcode (me_cod, me_iso, me_bez)
    SELECT 499, 'RO', 'Rolle'
    WHERE NOT EXISTS (SELECT 1 FROM mgcode WHERE me_iso = 'RO');
    GET DIAGNOSTICS v_rows = ROW_COUNT;
    c_me := c_me + v_rows;

    --------------------------------------------------------------------
    -- Hauptartikel anlegen
    --------------------------------------------------------------------
    INSERT INTO art (ak_ac, ak_nr, ak_bez, ak_standard_mgc, insert_date, modified_date, ak_los, ak_gewicht)
    SELECT
        'S' AS ak_ac,
        artikelnummer,
        COALESCE(text, 'keine Bezeichnung ') AS ak_bez,
        1,
        nullif(datum, '')::date,
        nullif(letzte_aenderung, '')::timestamp,
        losgroesse,
        REPLACE(REPLACE(nullif(gewicht_cu, ''), 'kg', ''), ',', '.')::NUMERIC
    FROM (
        SELECT *
        FROM jsonb_to_record((
                SELECT parsed_json -> 'kopf'
                FROM x_950_import.pdf_json_temp
                WHERE filename = p_filename
            )) AS x(
                text             text,
                datum            text,
                spannung         text,
                gewicht_cu       text,
                losgroesse       int,
                artikelnummer    text,
                letzte_aenderung text
            )
    ) AS sub
    WHERE NOT EXISTS (SELECT 1 FROM art WHERE ak_nr = sub.artikelnummer);
    GET DIAGNOSTICS v_rows = ROW_COUNT;
    c_art_haupt := c_art_haupt + v_rows;

    --------------------------------------------------------------------
    -- Standard-Mengeneinheit für Hauptartikel (artmgc)
    --------------------------------------------------------------------
    INSERT INTO artmgc (m_ak_nr, m_mgcode)
    SELECT artikelnummer, 1 AS m_mgcode
    FROM (
        SELECT *
        FROM jsonb_to_record((
                SELECT parsed_json -> 'kopf'
                FROM x_950_import.pdf_json_temp
                WHERE filename = p_filename
            )) AS x(
                text             text,
                datum            text,
                spannung         text,
                gewicht_cu       text,
                losgroesse       int,
                artikelnummer    text,
                letzte_aenderung text
            )
    ) AS sub
    WHERE NOT EXISTS (SELECT 1 FROM artmgc WHERE m_ak_nr = sub.artikelnummer);
    GET DIAGNOSTICS v_rows = ROW_COUNT;
    c_artmgc_std := c_artmgc_std + v_rows;

    --------------------------------------------------------------------
    -- Fehlende Positionsartikel anlegen
    --------------------------------------------------------------------
    INSERT INTO art (ak_ac, ak_nr, ak_bez, ak_standard_mgc, ak_txt)
    SELECT
        'S' AS ak_ac,
        artikelnummer,
        SUBSTRING(beschreibung, 1, 100),
        mecod,
        zusatzinfo
    FROM (
        SELECT *, me_cod AS mecod
        FROM jsonb_to_recordset((
                SELECT parsed_json -> 'positionen'
                FROM x_950_import.pdf_json_temp
                WHERE filename = p_filename
            )) AS x(
                menge         numeric,
                position      int,
                zusatzinfo    text,
                beschreibung  text,
                artikelnummer varchar,
                mengeneinheit text
            )
        JOIN LATERAL x_950_import.chk__me__translate(mengeneinheit) AS meiso ON true
        LEFT JOIN mgcode ON mgcode.me_iso = meiso
    ) AS sub
    WHERE NOT EXISTS (SELECT 1 FROM art WHERE ak_nr = sub.artikelnummer);
    GET DIAGNOSTICS v_rows = ROW_COUNT;
    c_art_pos := c_art_pos + v_rows;

    --------------------------------------------------------------------
    -- artmgc für Positionsartikel
    --------------------------------------------------------------------
    INSERT INTO artmgc (m_ak_nr, m_mgcode)
    SELECT artikelnummer, mecod
    FROM (
        SELECT *, me_cod AS mecod
        FROM jsonb_to_recordset((
                SELECT parsed_json -> 'positionen'
                FROM x_950_import.pdf_json_temp
                WHERE filename = p_filename
            )) AS x(
                menge         numeric,
                position      int,
                zusatzinfo    text,
                beschreibung  text,
                artikelnummer varchar,
                mengeneinheit text
            )
        JOIN LATERAL x_950_import.chk__me__translate(mengeneinheit) AS meiso ON true
        LEFT JOIN mgcode ON mgcode.me_iso = meiso
    ) AS sub
    WHERE NOT EXISTS (SELECT 1 FROM artmgc WHERE m_ak_nr = sub.artikelnummer);
    GET DIAGNOSTICS v_rows = ROW_COUNT;
    c_artmgc_pos := c_artmgc_pos + v_rows;

    --------------------------------------------------------------------
    -- Stückliste (stv)
    --------------------------------------------------------------------
    INSERT INTO stv (st_zn, st_n, st_pos, st_mgc, st_m)
    SELECT *
    FROM (
        SELECT
            (
                SELECT artikelnummer
                FROM (
                    SELECT *
                    FROM jsonb_to_record((
                            SELECT parsed_json -> 'kopf'
                            FROM x_950_import.pdf_json_temp
                            WHERE filename = p_filename
                        )) AS x(
                            text             text,
                            datum            text,
                            spannung         text,
                            gewicht_cu       text,
                            losgroesse       int,
                            artikelnummer    text,
                            letzte_aenderung text
                        )
                ) AS sub2
            )::varchar AS stzn,
            artikelnummer AS st_n,
            position      AS st_pos,
            m_id          AS st_mgc,
            menge         AS st_m
        FROM (
            SELECT menge, position, artikelnummer, m_id
            FROM jsonb_to_recordset((
                    SELECT parsed_json -> 'positionen'
                    FROM x_950_import.pdf_json_temp
                    WHERE filename = p_filename
                )) AS x(
                    menge         numeric,
                    position      int,
                    zusatzinfo    text,
                    beschreibung  text,
                    artikelnummer varchar,
                    mengeneinheit text
                )
            JOIN LATERAL x_950_import.chk__me__translate(mengeneinheit) AS meiso ON true
            LEFT JOIN mgcode ON mgcode.me_iso = meiso
            LEFT JOIN artmgc ON m_ak_nr = artikelnummer
        ) AS sub
    ) AS sub1
    WHERE NOT EXISTS (
        SELECT 1
        FROM stv
        WHERE st_n  = sub1.st_n
          AND st_zn = sub1.stzn
    );
    GET DIAGNOSTICS v_rows = ROW_COUNT;
    c_stv := c_stv + v_rows;

    --------------------------------------------------------------------
    -- Aufsummieren & Temp löschen (optional)
    --------------------------------------------------------------------
    c_art    := c_art_haupt + c_art_pos;
    c_artmgc := c_artmgc_std + c_artmgc_pos;

    DELETE FROM x_950_import.pdf_json_temp
    WHERE filename = p_filename;
    GET DIAGNOSTICS c_del = ROW_COUNT;

    --------------------------------------------------------------------
    -- Erfolg-JSON zurückgeben
    --------------------------------------------------------------------
    RETURN json_build_object(
      'success', true,
      'filename', p_filename,
      'inserted', json_build_object(
        'mgcode', c_me,
        'art', c_art,
        'art_haupt', c_art_haupt,
        'art_pos', c_art_pos,
        'artmgc', c_artmgc,
        'artmgc_std', c_artmgc_std,
        'artmgc_pos', c_artmgc_pos,
        'stv', c_stv
      ),
      'deleted_temp', c_del
    );

  /*EXCEPTION
    WHEN OTHERS THEN
      RETURN json_build_object(
        'success', false,
        'filename', p_filename,
        'error', SQLERRM
      );*/
  END $$;


CREATE OR REPLACE FUNCTION api.chk_opl_import_json__from_x_950_import(
    p_filename text
  )
  RETURNS json
  LANGUAGE plpgsql
  AS $BODY$
  DECLARE
    v_head jsonb;
    v_ops  jsonb;
    _ak_nr text;
    _op_ix bigint;

    c_art     int := 0;
    c_artmgc  int := 0;
    c_opl     int := 0;
    c_ksv     int := 0;
    c_op2     int := 0;
    c_del     int := 0;
  BEGIN
    -- Daten laden
    SELECT t.parsed_json->'kopf', t.parsed_json->'arbeitsgaenge'
      INTO v_head, v_ops
    FROM x_950_import.pdf_json_temp t
    WHERE t.filename = p_filename;

    IF v_head IS NULL THEN
      RAISE EXCEPTION 'Kein Eintrag in x_950_import.pdf_json_temp für %', p_filename
        USING ERRCODE = 'no_data_found';
    END IF;

    -- artikelnummer aus Kopf
    SELECT artikelnummer
      INTO _ak_nr
    FROM jsonb_to_record(v_head) AS x (
      dlz               INT,
      text              TEXT,
      spannung          TEXT,
      losgroesse        INT,
      zusatzinfo        TEXT,
      bezeichnung       TEXT,
      artikelnummer     TEXT
    );

    IF _ak_nr IS NULL OR btrim(_ak_nr) = '' THEN
      RAISE EXCEPTION 'artikelnummer fehlt im Kopf (%).', p_filename;
    END IF;

    -- Hauptartikel anlegen
    INSERT INTO art ( ak_ac, ak_nr, ak_bez, ak_standard_mgc, insert_by, modified_by )
    SELECT 'S' AS ak_ac,
          x.artikelnummer,
          COALESCE(x.text, 'keine Bezeichnung ') AS ak_bez,
          1,
          current_user,
          current_user
    FROM jsonb_to_record(v_head) AS x (
      dlz               INT,
      text              TEXT,
      spannung          TEXT,
      losgroesse        INT,
      zusatzinfo        TEXT,
      bezeichnung       TEXT,
      artikelnummer     TEXT
    )
    WHERE NOT EXISTS (SELECT 1 FROM art a WHERE a.ak_nr = x.artikelnummer);
    GET DIAGNOSTICS c_art = ROW_COUNT;

    -- artmgc anlegen
    INSERT INTO artmgc ( m_ak_nr, m_mgcode )
    SELECT x.artikelnummer, 1
    FROM jsonb_to_record(v_head) AS x (
      dlz               INT,
      text              TEXT,
      spannung          TEXT,
      losgroesse        INT,
      zusatzinfo        TEXT,
      bezeichnung       TEXT,
      artikelnummer     TEXT
    )
    WHERE NOT EXISTS (SELECT 1 FROM artmgc m WHERE m.m_ak_nr = x.artikelnummer);
    GET DIAGNOSTICS c_artmgc = ROW_COUNT;

    -- AVOR (OPL) anlegen
    INSERT INTO opl ( op_n, op_vi, op_lg, op_txt, modified_by )
    SELECT x.artikelnummer, '1', x.losgroesse, x.text, nullif(x.letzte_aenderung, '')
    FROM jsonb_to_record(v_head) AS x (
      dlz               INT,
      text              TEXT,
      spannung          TEXT,
      losgroesse        INT,
      zusatzinfo        TEXT,
      bezeichnung       TEXT,
      artikelnummer     TEXT,
      benutzung_bis     text,
      letzte_aenderung  text
    )
    WHERE NOT EXISTS (SELECT 1 FROM opl o WHERE o.op_n = UPPER(x.artikelnummer))
    RETURNING op_ix INTO _op_ix;
    GET DIAGNOSTICS c_opl = ROW_COUNT;

    -- Falls bereits vorhanden, die existierende op_ix bestimmen
    IF _op_ix IS NULL THEN
      SELECT o.op_ix
        INTO _op_ix
      FROM opl o
      WHERE o.op_n = UPPER(_ak_nr)
      ORDER BY o.op_ix
      LIMIT 1;
    END IF;

    -- Kostenstellen (ksv) anlegen
    INSERT INTO ksv ( ks_abt )
    SELECT DISTINCT a.fgr
    FROM jsonb_to_recordset(v_ops) AS a (
      lf                TEXT,
      sa                TEXT,
      sp                TEXT,
      tr                TEXT,
      fgr               TEXT,
      vlz               TEXT,
      t_nr              TEXT,
      te_me             TEXT,
      status            TEXT,
      details           TEXT,
      position          INT,
      beschreibung      TEXT,
      letzte_aenderung  text
    )
    WHERE a.fgr IS NOT NULL AND a.fgr <> ''
      AND NOT EXISTS (SELECT 1 FROM ksv k WHERE k.ks_abt = UPPER(a.fgr));
    GET DIAGNOSTICS c_ksv = ROW_COUNT;

    -- Arbeitspläne (op2) anlegen
    INSERT INTO op2 ( o2_ix, o2_n, o2_txt, o2_ks, o2_th, o2_tr, modified_by )
    SELECT _op_ix,
          a.position,
          concat_ws(E'\n\n', a.beschreibung, a.details),
          a.fgr,
          coalesce(a.te_me, 0),
          coalesce(a.tr/60, 0),
          nullif(a.letzte_aenderung, '')
    FROM jsonb_to_recordset(v_ops) AS a (
      lf                TEXT,
      sa                TEXT,
      sp                TEXT,
      tr                numeric,
      fgr               TEXT,
      vlz               TEXT,
      t_nr              TEXT,
      te_me             numeric,
      status            TEXT,
      details           TEXT,
      position          INT,
      beschreibung      TEXT,
      letzte_aenderung  text
    )
    WHERE NOT EXISTS (
      SELECT 1 FROM op2 o WHERE o.o2_ix = _op_ix AND o.o2_n = a.position
    );
    GET DIAGNOSTICS c_op2 = ROW_COUNT;

    -- Temp-Eintrag löschen
    DELETE FROM x_950_import.pdf_json_temp t WHERE t.filename = p_filename;
    GET DIAGNOSTICS c_del = ROW_COUNT;

    RETURN json_build_object(
      'success', true,
      'filename', p_filename,
      'ak_nr', _ak_nr,
      'op_ix', _op_ix,
      'inserted', json_build_object(
        'art', c_art,
        'artmgc', c_artmgc,
        'opl', c_opl,
        'ksv', c_ksv,
        'op2', c_op2
      ),
      'deleted_temp', c_del
    );
  EXCEPTION
    WHEN OTHERS THEN
      RETURN json_build_object(
        'success', false,
        'filename', p_filename,
        'error', SQLERRM
      );
  END $BODY$;


CREATE OR REPLACE FUNCTION z_50_customer.stv_extract__from_pdf__to_json(
    pdf_data bytea
  )
  RETURNS jsonb
  LANGUAGE plpython3u
  AS $BODY$
import os
import json
import urllib.request

# OpenAI API-Key – am besten als Umgebungsvariable setzen!
OPENAI_API_KEY = plpy.execute("SELECT tsystem.settings__get('OpenAI_API_Key')")[0]["settings__get"]
OPENAI_API_URL = "https://api.openai.com/v1/chat/completions"

# JSON Schema
json_schema = {
  "name": "stueckliste",
  "schema": {
    "type": "object",
    "properties": {
      "kopf": {
        "type": "object",
        "properties": {
          "artikelnummer": {
            "type": "string",
            "description": "Die Artikelnummer des Oberelements."
          },
          "text": {
            "type": "string",
            "description": "Bezeichnung oder Beschreibung des Oberelements."
          },
          "datum": {
            "type": "string",
            "description": "Datum des Dokuments."
          },
          "letzte_aenderung": {
            "type": "string",
            "description": "Datum der letzten Änderung."
          },
          "losgroesse": {
            "type": "number",
            "description": "Größe der Losgröße."
          },
          "spannung": {
            "type": "string",
            "description": "Spannung des Oberelements."
          },
          "gewicht_cu": {
            "type": "string",
            "description": "Gewicht des Oberelements in Kupfer."
          }
        },
        "required": [
          "artikelnummer",
          "text",
          "datum",
          "letzte_aenderung",
          "losgroesse",
          "spannung",
          "gewicht_cu"
        ],
        "additionalProperties": False
      },
      "positionen": {
        "type": "array",
        "description": "Liste der Positionen in der Stückliste.",
        "items": {
          "type": "object",
          "properties": {
            "position": {
              "type": "number",
              "description": "Die Positionsnummer der Artikel."
            },
            "artikelnummer": {
              "type": "string",
              "description": "Artikelnummer des Einzelteils."
            },
            "beschreibung": {
              "type": "string",
              "description": "Beschreibung des Einzelteils."
            },
            "menge": {
              "type": "number",
              "description": "Menge des Einzelteils."
            },
            "mengeneinheit": {
              "type": "string",
              "description": "Einheit der Menge (z.B. Stück, kg)."
            },
            "zusatzinfo": {
              "type": "string",
              "description": "Zusätzliche Informationen zum Einzelteil."
            }
          },
          "required": [
            "position",
            "artikelnummer",
            "beschreibung",
            "menge",
            "mengeneinheit",
            "zusatzinfo"
          ],
          "additionalProperties": False
        }
      }
    },
    "required": [
      "kopf",
      "positionen"
    ],
    "additionalProperties": False
  },
  "strict": True
}

# Prompt wie besprochen
prompt = "Extrahiere aus dem PDF die Stückliste"

# Das PDF aus bytea in base64 umwandeln
import base64
pdf_bytes = bytes(pdf_data)
pdf_base64 = base64.b64encode(pdf_bytes).decode('utf-8')
pdf_data_url = f"data:application/pdf;base64,{pdf_base64}"

# OpenAI expects 'data' and 'mime_type' for files (vision)
user_message = [
    {
        "type": "file",
        "file": {
            "filename": "stueckliste.pdf",
            "file_data": pdf_data_url
        }
    },
    {
        "type": "text",
        "text": prompt
    }
]

payload = {
    "model": "gpt-4o-mini",  # Oder gpt-4-vision-preview
    "messages": [
        {"role": "system", "content": "Du bist ein Assistent für die strukturierte Datenextraktion."},
        {"role": "user", "content": user_message}
    ],
    "response_format": {
        "type": "json_schema",
        "json_schema": json_schema
    },
    "max_tokens": 4096
}

headers = {
    "Content-Type": "application/json",
    "Authorization": f"Bearer {OPENAI_API_KEY}"
}

# plpy.notice(json.dumps(payload).encode('utf-8'))

req = urllib.request.Request(
    OPENAI_API_URL,
    data=json.dumps(payload).encode('utf-8'),
    headers=headers
)

# plpy.notice(headers)

try:
    with urllib.request.urlopen(req) as response:
        result = response.read()
        result_json = json.loads(result)
except Exception as e:
    plpy.notice(e.read().decode('utf-8'))
    return {"error": str(e)}

try:
    # Die Antwort von OpenAI enthält das JSON als Text
    ai_answer = result_json['choices'][0]['message']['content']
    plpy.notice("ai_answer")
    plpy.notice(ai_answer)
    # out_json = json.loads(ai_answer)
    return ai_answer
except Exception as e:
    return {"error": "JSON parse error", "answer": result_json.get('choices', [{}])[0].get('message', {}).get('content', ''), "exception": str(e)}
$BODY$;


CREATE OR REPLACE FUNCTION z_50_customer.opl_extract__from_pdf__to_json(
    pdf_data bytea
  )
  RETURNS jsonb
  LANGUAGE plpython3u
  AS $BODY$
import os
import json
import urllib.request

# OpenAI API-Key – am besten als Umgebungsvariable setzen!
OPENAI_API_KEY = plpy.execute("SELECT tsystem.settings__get('OpenAI_API_Key')")[0]["settings__get"]
OPENAI_API_URL = "https://api.openai.com/v1/chat/completions"

# JSON Schema
json_schema = {
    "name": "arbeitsplan",
    "schema": {
        "type": "object",
        "properties": {
            "kopf": {
                "type": "object",
                "properties": {
                    "artikelnummer": {
                        "type": "string",
                        "description": "Artikelnummer des Produkts, für das der Arbeitsplan gilt."
                    },
                    "text": {
                        "type": "string",
                        "description": "Bezeichnung oder Beschreibung des Produkts."
                    },
                    "letzte_aenderung": {
                        "type": "string",
                        "description": "Datum der letzten Änderung des Arbeitsplans."
                    },
                    "losgroesse": {
                        "type": ["number", "null"],
                        "description": "Größe der Losgröße (falls nicht vorhanden: null)."
                    },
                    "spannung": {
                        "type": ["string", "null"],
                        "description": "Angabe zur Spannung bzw. technischen Parametern (falls nicht vorhanden: null)."
                    },
                    "benutzung_bis": {
                        "type": ["string", "null"],
                        "description": "Letztes Benutzungsdatum (falls nicht vorhanden: null)."
                    },
                    "dlz": {
                        "type": ["number", "null"],
                        "description": "Durchlaufzeit (falls nicht vorhanden: null)."
                    },
                    "bezeichnung": {
                        "type": ["string", "null"],
                        "description": "Weitere Bezeichnung, falls vorhanden, sonst null."
                    },
                    "zusatzinfo": {
                        "type": ["string", "null"],
                        "description": "Zusätzliche Angaben (z.B. Kupfergewicht, Index etc., falls nicht vorhanden: null)."
                    }
                },
                "required": [
                    "artikelnummer",
                    "text",
                    "letzte_aenderung",
                    "losgroesse",
                    "spannung",
                    "benutzung_bis",
                    "dlz",
                    "bezeichnung",
                    "zusatzinfo"
                ],
                "additionalProperties": False
            },
            "arbeitsgaenge": {
                "type": "array",
                "description": "Liste der Arbeitsschritte bzw. Arbeitsgänge.",
                "items": {
                    "type": "object",
                    "properties": {
                        "position": {
                            "type": "number",
                            "description": "Positionsnummer des Arbeitsgangs."
                        },
                        "fgr": {
                            "type": ["string", "null"],
                            "description": "Fertigungsgruppe (FGR, falls nicht vorhanden: null)."
                        },
                        "sa": {
                            "type": ["string", "null"],
                            "description": "Sammelarbeitsgang (SA, falls nicht vorhanden: null)."
                        },
                        "beschreibung": {
                            "type": "string",
                            "description": "Kurzbeschreibung des Arbeitsgangs."
                        },
                        "t_nr": {
                            "type": ["string", "null"],
                            "description": "Tätigkeitsnummer (T-Nr, falls nicht vorhanden: null)."
                        },
                        "sp": {
                            "type": ["string", "null"],
                            "description": "Sonderkennzeichen/Prüfmerkmal (SP, falls nicht vorhanden: null)."
                        },
                        "te_me": {
                            "type": ["number", "null"],
                            "description": "TE/ME, Bearbeitungszeit pro Mengeneinheit (falls nicht vorhanden: null)."
                        },
                        "tr": {
                            "type": ["number", "null"],
                            "description": "TR (z.B. Rüstzeit, falls nicht vorhanden: null)."
                        },
                        "vlz": {
                            "type": ["string", "null"],
                            "description": "Verlängerte Laufzeit (VLZ, falls nicht vorhanden: null)."
                        },
                        "lf": {
                            "type": ["string", "null"],
                            "description": "LF (z.B. Losfaktor, falls nicht vorhanden: null)."
                        },
                        "status": {
                            "type": ["string", "null"],
                            "description": "Status oder Freigabe (z.B. A, falls nicht vorhanden: null)."
                        },
                        "letzte_aenderung": {
                            "type": ["string", "null"],
                            "description": "Letztes Änderungsdatum des Arbeitsgangs (falls nicht vorhanden: null)."
                        },
                        "details": {
                            "type": ["string", "null"],
                            "description": "Mehrzeilige Detailbeschreibung des Arbeitsschritts (falls nicht vorhanden: null)."
                        }
                    },
                    "required": [
                        "position",
                        "fgr",
                        "sa",
                        "beschreibung",
                        "t_nr",
                        "sp",
                        "te_me",
                        "tr",
                        "vlz",
                        "lf",
                        "status",
                        "letzte_aenderung",
                        "details"
                    ],
                    "additionalProperties": False
                }
            }
        },
        "required": [
            "kopf",
            "arbeitsgaenge"
        ],
        "additionalProperties": False
    },
    "strict": True
}

# Prompt wie besprochen
prompt = "Extrahiere aus dem PDF den Arbeitsplan"

# Das PDF aus bytea in base64 umwandeln
import base64
pdf_bytes = bytes(pdf_data)
pdf_base64 = base64.b64encode(pdf_bytes).decode('utf-8')
pdf_data_url = f"data:application/pdf;base64,{pdf_base64}"

# OpenAI expects 'data' and 'mime_type' for files (vision)
user_message = [
    {
        "type": "file",
        "file": {
            "filename": "arbeitsplan.pdf",
            "file_data": pdf_data_url
        }
    },
    {
        "type": "text",
        "text": prompt
    }
]

payload = {
    "model": "gpt-4o-mini",  # Oder gpt-4-vision-preview
    "messages": [
        {"role": "system", "content": "Du bist ein Assistent für die strukturierte Datenextraktion."},
        {"role": "user", "content": user_message}
    ],
    "response_format": {
        "type": "json_schema",
        "json_schema": json_schema
    },
    "max_tokens": 4096
}

headers = {
    "Content-Type": "application/json",
    "Authorization": f"Bearer {OPENAI_API_KEY}"
}

# plpy.notice(json.dumps(payload).encode('utf-8'))

req = urllib.request.Request(
    OPENAI_API_URL,
    data=json.dumps(payload).encode('utf-8'),
    headers=headers
)

# plpy.notice(headers)

try:
    with urllib.request.urlopen(req) as response:
        result = response.read()
        result_json = json.loads(result)
except Exception as e:
    plpy.notice(e.read().decode('utf-8'))
    return {"error": str(e)}

try:
    # Die Antwort von OpenAI enthält das JSON als Text
    ai_answer = result_json['choices'][0]['message']['content']
    plpy.notice("ai_answer")
    plpy.notice(ai_answer)
    # out_json = json.loads(ai_answer)
    return ai_answer
except Exception as e:
    return {"error": "JSON parse error", "answer": result_json.get('choices', [{}])[0].get('message', {}).get('content', ''), "exception": str(e)}
$BODY$;


CREATE OR REPLACE FUNCTION api.stv_extract__from_pdf__to_json(
    pdf_data bytea
  )
  RETURNS jsonb
  LANGUAGE sql
  AS $BODY$
    SELECT z_50_customer.stv_extract__from_pdf__to_json(pdf_data)
  $BODY$;

CREATE OR REPLACE FUNCTION api.opl_extract__from_pdf__to_json(
    pdf_data bytea
  )
  RETURNS jsonb
  LANGUAGE sql
  AS $BODY$
    SELECT z_50_customer.opl_extract__from_pdf__to_json(pdf_data)
  $BODY$;

CREATE TABLE x_950_import.pdf_json_temp
  (
    filename    text NOT NULL PRIMARY KEY,
    parsed_json jsonb
  );

CREATE OR REPLACE FUNCTION api.chk_ask_import()
  RETURNS "text/html"
  LANGUAGE sql
  AS $BODY$
  SELECT $html$

<!DOCTYPE html>
<html lang="de">
<head>
  <meta charset="UTF-8">
  <title>Auftragsdaten: EVGT 3500VA</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <script type="module">
    import { getCookie, setupAuth, fetchWithAuth } from './file?path=auth-header.js';
    setupAuth(); // verwendet in der config im Standard => cookieName: 'jwt', headerName: 'Authorization', prefix: 'Bearer'
    window.fetchWithAuth = fetchWithAuth;
    window.dispatchEvent(new Event('auth-ready'));
  </script>
  <style>
    :root{
      --bg:#f6f7fa; --card:#ffffff; --text:#1d2a3a; --muted:#6b7280;
      --brand:#2a9d8f; --brand-2:#457b9d; --danger:#b22; --ok:#217a2a;
      --border:#e5e7eb; --shadow:0 6px 20px rgba(0,0,0,.06);
      --radius:12px; --radius-sm:8px;
      --pad:2em; --gap:16px; --gap-sm:10px;
    }
    *{box-sizing:border-box}
    html,body{height:100%}
    body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Helvetica,Arial,sans-serif;background:var(--bg);margin:0;padding:0 6vw;color:var(--text)}
    h1{color:#264653;margin:24px 0}
    h2{margin:0 0 12px 0}
    .card{background:var(--card);border-radius:var(--radius);box-shadow:var(--shadow);margin:0 0 24px 0;padding:var(--pad)}
    .grid{display:grid;grid-template-columns:repeat(12,1fr);gap:var(--gap)}
    .grid-2col{display:grid;grid-template-columns:1fr 1fr;gap:var(--gap)}
    @media (max-width: 900px){ body{padding:0 4vw} .grid-2col{grid-template-columns:1fr} }
    table{width:100%;border-collapse:separate;border-spacing:0}
    th,td{padding:10px;border-bottom:1px solid var(--border);vertical-align:top}
    th{background:#f3f6fb;text-align:left;position:sticky;top:0;z-index:1}
    tr:nth-child(even) td{background:#fafbfd}
    tr:hover td{background:#f5fbfa}
    .table-wrap{overflow:auto;border:1px solid var(--border);border-radius:var(--radius)}
    .small{font-size:.9em;color:var(--muted)}
    .muted{color:var(--muted)}
    .error{color:var(--danger);font-weight:600;margin:.5em 0}
    .success{color:var(--ok);font-weight:600}
    .loading{color:var(--muted);margin:.5em 0}

    label{display:block;font-size:.9em;color:#5b6775;margin-bottom:4px}
    input,textarea,select{width:100%;font-size:1em;border-radius:var(--radius-sm);border:1px solid #cfd8e3;padding:8px 10px;background:#fff;transition:border .15s, box-shadow .15s}
    input:focus,textarea:focus,select:focus{outline:none;border-color:var(--brand);box-shadow:0 0 0 3px rgba(42,157,143,.18)}
    textarea{resize:vertical;min-height:2.5em}

    .btn{display:inline-flex;align-items:center;gap:8px;border:none;border-radius:10px;padding:.65em 1.2em;font-weight:600;cursor:pointer}
    .btn:disabled{opacity:.6;cursor:not-allowed}
    .btn-primary{background:var(--brand);color:#fff}
    .btn-secondary{background:var(--brand-2);color:#fff}
    .btn-outline{background:#f8fafc;border:1px solid #cfd8e3;color:#1f2937}
    .btn-ghost{background:transparent;color:#1f2937}
    .chip{display:inline-block;padding:.25em .6em;border-radius:999px;border:1px solid var(--border);background:#fff}

    /* Drag & Drop */
    .dropzone{border:2px dashed #9aa6b2;border-radius:var(--radius);background:#f9fbff;padding:1.5em;text-align:center;transition:.15s;cursor:pointer}
    .dropzone.dragover{border-color:var(--brand);background:#eef9f7}
    .dropzone .hint{color:var(--muted)}
    .dropzone input[type=file]{display:none}

    /* Actions bar sticky bottom */
    .actions-bar{position:sticky;bottom:0;z-index:5;margin-top:20px;background:linear-gradient(180deg, rgba(246,247,250,0) 0%, var(--bg) 35%);padding:12px 0}
    .actions{display:flex;gap:10px;flex-wrap:wrap;align-items:center}

    /* Toasts */
    .toasts{position:fixed;right:20px;bottom:20px;display:flex;flex-direction:column;gap:10px;z-index:9999}
    .toast{padding:.7em 1em;border-radius:10px;box-shadow:var(--shadow);background:#ffffffd9;border:1px solid var(--border)}
    .toast.ok{border-color:#bfe7d8}
    .toast.err{border-color:#f2b8b5}

    /* Utility widths for number cols */
    .w-80{width:80px} .w-100{width:100px} .w-140{width:140px}
    .nowrap{white-space:nowrap}
    .sr-only{position:absolute!important;height:1px;width:1px;overflow:hidden;clip:rect(1px,1px,1px,1px);white-space:nowrap;border:0;padding:0;margin:-1px}
    .spinner{inline-size:1em;block-size:1em;border-radius:50%;border:.2em solid #cfe3df;border-top-color:var(--brand);animation:spin 1s linear infinite}
    @keyframes spin{to{transform:rotate(360deg)}}
  </style>
</head>
<body>
  <h1>ASK Import</h1>

  <!-- Upload -->
  <div class="flex">
  <section class="card" aria-labelledby="upload-h">
    <h2 id="upload-h">Arbeitsplan als PDF hochladen &amp; extrahieren</h2>
    <div id="upload-status" class="small" aria-live="polite"></div>
    <div id="dropzone" class="dropzone" tabindex="0" role="button" aria-label="PDF hier ablegen oder klicken, um eine Datei zu wählen">
      <div><strong>Datei hierher ziehen</strong> oder klicken, um eine PDF zu wählen.</div>
      <div class="hint">Erlaubt: *.pdf &nbsp;•&nbsp; Upload an <code>./opl_extract__from_pdf__to_json</code></div>
      <input id="file-input" type="file" accept="application/pdf,.pdf" aria-label="PDF auswählen">
    </div>
    <div class="small muted" id="picked-file" aria-live="polite"></div>
    <div style="margin-top:10px;">
      <button type="button" id="btn-extract" class="btn btn-outline">
        <span class="sr-only">Extrahieren &amp; speichern</span>Extrahieren &amp; speichern
      </button>
      <span class="small chip" id="upload-hint">Ergebnis wird in <code>/x_950_import/pdf_json_temp</code> gespeichert.</span>
    </div>
    <div id="upload-error" class="error" style="display:none" aria-live="assertive"></div>
    <div id="upload-success" class="success" style="display:none" aria-live="polite"></div>
  </section>

  <!-- Datei wählen -->
  <section class="card" aria-labelledby="datei-h">
    <h2 id="datei-h">Datei wählen</h2>
    <div class="grid-2col">
      <div>
        <label for="src_filename">Dateiname (Quelle)</label>
        <input id="src_filename" name="src_filename" list="filevorschlaege" placeholder="14409-G-Arbeitsplan.pdf" autocomplete="off">
        <datalist id="filevorschlaege"></datalist>
        <div class="muted small" id="vorschlaege-info" aria-live="polite"></div>
      </div>
      <div>
        <label for="dst_filename">Dateiname (Ziel)</label>
        <input id="dst_filename" name="dst_filename" placeholder="14409-G-Arbeitsplan.pdf">
      </div>
      <div class="nowrap">
        <button type="button" id="btn-load" class="btn btn-primary">Laden</button>
        <button type="button" id="btn-refresh-suggest" class="btn btn-outline">Vorschläge aktualisieren</button>
      </div>
    </div>
  </section>
  </div>

  <!-- Formular -->
  <form id="auftragsForm" autocomplete="off" onsubmit="event.preventDefault();saveForm();" aria-labelledby="form-h">
    <section class="card" aria-labelledby="kopf-h">
      <h2 id="kopf-h">Kopfdaten</h2>
      <div class="grid-2col">
        <div><label for="artikelnummer">Artikelnummer</label><input id="artikelnummer" name="artikelnummer" required></div>
        <div><label for="bezeichnung">Bezeichnung</label><input id="bezeichnung" name="bezeichnung"></div>
        <div><label for="text">Text</label><input id="text" name="text"></div>
        <div><label for="spannung">Spannung</label><input id="spannung" name="spannung"></div>
        <div><label for="losgroesse">Losgröße</label><input id="losgroesse" name="losgroesse" type="number" min="0" step="1" inputmode="numeric"></div>
        <div><label for="zusatzinfo">Cu-Gewicht</label><input id="zusatzinfo" name="zusatzinfo"></div>
        <div><label for="benutzung_bis">Benutzung bis</label><input id="benutzung_bis" name="benutzung_bis" type="date"></div>
        <div><label for="letzte_aenderung">Letzte Änderung</label><input id="letzte_aenderung" name="letzte_aenderung" type="datetime-local"></div>
      </div>
    </section>

    <section class="card" aria-labelledby="ag-h">
      <div class="grid" style="grid-template-columns:1fr auto;align-items:end">
        <h2 id="ag-h" style="grid-column:1/2;margin:0">Arbeitsgänge</h2>
        <div class="small" id="ag-count" style="justify-self:end"></div>
        <div style="grid-column:1/-1;margin-top:10px;display:flex;gap:10px;align-items:center">
          <label for="ag-filter" class="small">Schnellfilter</label>
          <input id="ag-filter" placeholder="z. B. Beschreibung oder FGR" style="max-width:320px">
          <span class="small muted">Enter löscht Filter</span>
        </div>
      </div>

      <div id="loading" class="loading" aria-live="polite">Lade Daten …</div>
      <div id="error" class="error" style="display:none" aria-live="assertive"></div>

      <div class="table-wrap" id="ag-wrap" style="display:none">
        <table id="ag-table">
          <thead>
            <tr>
              <th class="w-80">Pos</th>
              <th>Beschreibung</th>
              <th class="w-100">Fertigungszeit [min/ME]</th>
              <th class="w-100">Rüstzeit [h]</th>
              <th class="w-140">Fertigungsgruppe</th>
              <th class="w-100">Status</th>
              <th>Details</th>
              <th class="w-140">Letzte Änderung</th>
            </tr>
          </thead>
          <tbody id="ag-tbody"></tbody>
        </table>
      </div>
    </section>

    <div class="actions-bar">
      <div class="actions">
        <button type="submit" id="btn-save" class="btn btn-primary">Daten speichern</button>
        <button type="button" id="btn-import" class="btn btn-secondary">In ERP importieren</button>
        <button type="button" id="btn-reload" class="btn btn-ghost">Neu laden</button>
        <span id="save-result" class="small" aria-live="polite"></span>
        <span id="import-result" class="small" aria-live="polite"></span>
      </div>
    </div>
  </form>

  <div class="toasts" aria-live="polite" aria-atomic="true"></div>

  <script>
    // --- Endpunkte (unverändert) ---
    const endpointBase     = "/pdf_json_temp";  // GET: ?filename=eq.<name>
    const saveUrl          = "/pdf_json_temp";  // POST: { filename, parsed_json }
    const extractEndpoint  = "./opl_extract__from_pdf__to_json"; // POST: { pdf_data: bytea-hex }
    const importUrl        = "./chk_opl_import_json__from_x_950_import"; // POST: { p_filename }

    // --- State ---
    const defaultSrcFilename = "";
    let currentSrcFilename = defaultSrcFilename;
    let kopfData = {};
    let arbeitsgaengeData = [];
    let filenameSuggestions = [];
    let pickedFile = null;
    let isDirty = false;

    // --- Helpers ---
    const $ = sel => document.querySelector(sel);
    const $$ = sel => Array.from(document.querySelectorAll(sel));
    function toast(msg, type='ok'){ const t = document.createElement('div'); t.className='toast '+(type==='err'?'err':'ok'); t.textContent=msg; const box=$('.toasts'); box.appendChild(t); setTimeout(()=>{ t.style.opacity='0'; setTimeout(()=>t.remove(),300) }, 4000); }
    function setBusy(btn, busy=true){ if(!btn)return; if(busy){ btn.disabled=true; const s=document.createElement('span'); s.className='spinner'; s.setAttribute('aria-hidden','true'); btn.__spinner=s; btn.appendChild(s);} else { btn.disabled=false; if(btn.__spinner){ btn.__spinner.remove(); btn.__spinner=null; } } }
    function autoResize(textarea){ textarea.style.height='auto'; textarea.style.height=(textarea.scrollHeight+2)+'px'; }
    window.addEventListener('beforeunload', e=>{ if(isDirty){ e.preventDefault(); e.returnValue=''; }});
    function clearLoadError() {const err = document.getElementById('error'); if (err) { err.textContent = ''; err.style.display = 'none'; }}

    // --- Upload & Extraktion ---
    const dz = $('#dropzone'); const fileInput = $('#file-input'); const btnExtract = $('#btn-extract');
    dz.addEventListener('click', ()=> fileInput.click());
    dz.addEventListener('keydown', e=>{ if(e.key==='Enter' || e.key===' '){ e.preventDefault(); fileInput.click(); }});
    dz.addEventListener('dragover', e=>{ e.preventDefault(); dz.classList.add('dragover'); });
    dz.addEventListener('dragleave', ()=> dz.classList.remove('dragover'));
    dz.addEventListener('drop', e=>{
      e.preventDefault(); dz.classList.remove('dragover');
      if(e.dataTransfer.files && e.dataTransfer.files[0]) handlePickedFile(e.dataTransfer.files[0]);
    });
    fileInput.addEventListener('change', ()=>{ if(fileInput.files && fileInput.files[0]) handlePickedFile(fileInput.files[0]); });
    function handlePickedFile(f){
      if(!/\.pdf$/i.test(f.name)){ showUploadError("Nur PDF-Dateien sind erlaubt."); return; }
      pickedFile = f; hideUploadMessages(); $('#picked-file').textContent = `Gewählt: ${f.name} (${Math.round(f.size/1024)} KB)`;
    }
    function hideUploadMessages(){ const e=$('#upload-error'), s=$('#upload-success'); e.style.display='none'; e.textContent=''; s.style.display='none'; s.textContent=''; }
    function showUploadError(msg){ const el=$('#upload-error'); el.style.display=''; el.textContent=msg; toast(msg,'err'); }
    function showUploadSuccess(msg){ const el=$('#upload-success'); el.style.display=''; el.textContent=msg; toast(msg,'ok'); }
    function arrayBufferToHex(ab){ const bytes=new Uint8Array(ab); const hex=Array.from(bytes,b=>b.toString(16).padStart(2,'0')).join(''); return '\\x'+hex; }

    btnExtract.addEventListener('click', async ()=>{
      hideUploadMessages(); const status=$('#upload-status'); if(!pickedFile){ showUploadError("Bitte zuerst eine PDF auswählen."); return; }
      try{
        setBusy(btnExtract,true);
        status.textContent="Lese Datei …";
        const buf = await pickedFile.arrayBuffer(); const hex = arrayBufferToHex(buf);
        status.textContent="Sende an Extraktions-Endpunkt …";
        const extractResp = await fetchWithAuth(extractEndpoint,{ method:"POST", headers:{ "Content-Type":"application/json","Prefer":"resolution=merge-duplicates" }, body: JSON.stringify({ pdf_data: hex }) });
        if(!extractResp.ok) throw new Error("Extraktion fehlgeschlagen (HTTP "+extractResp.status+")");
        let extractJson = await extractResp.json().catch(()=>null); if(!extractJson) throw new Error("Leere Antwort vom Extraktions-Endpunkt.");
        let parsed=null;
        if(extractJson.parsed_json) parsed=extractJson.parsed_json;
        else if(Array.isArray(extractJson) && extractJson[0]?.parsed_json) parsed=extractJson[0].parsed_json;
        else if(extractJson.kopf || extractJson.arbeitsgaenge) parsed=extractJson;
        if(!parsed) throw new Error("Antwort enthält kein erwartetes parsed_json.");
        status.textContent="Speichere Ergebnis …";
        const storeFilename = pickedFile.name;
        const saveResp = await fetchWithAuth(saveUrl,{ method:"POST", headers:{ "Content-Type":"application/json","Prefer":"resolution=merge-duplicates","Content-Profile":"x_950_import" }, body: JSON.stringify({ filename: storeFilename, parsed_json: parsed })});
        if(!saveResp.ok) throw new Error("Speichern fehlgeschlagen (HTTP "+saveResp.status+")");
        await loadFilenameSuggestions();
        $('#src_filename').value = storeFilename;
        if(!$('#dst_filename').dataset.touched) $('#dst_filename').value = deriveDstFilename(storeFilename);
        clearLoadError();
        fillKopf(parsed.kopf ?? {}); fillAG(parsed.arbeitsgaenge ?? []); $('#ag-wrap').style.display='';
        status.textContent=""; showUploadSuccess("Extraktion und Speicherung erfolgreich.");
      }catch(e){ showUploadError(e.message || String(e)); } finally{ setBusy(btnExtract,false); }
    });

    // --- Vorschläge & Dateiladen ---
    function buildUrl(fname){ return endpointBase + "?filename=eq." + encodeURIComponent(fname); }
    function deriveDstFilename(src){ return (src||""); } // bewusst: Ziel=Quelle
    function setInitialFilenames(){ const s=$('#src_filename'), d=$('#dst_filename'); if(!s.value) s.value=defaultSrcFilename; if(!d.value) d.value=deriveDstFilename(s.value); }
    function populateDatalist(options){ const dl=$('#filevorschlaege'); dl.innerHTML=''; options.forEach(v=>{ const opt=document.createElement('option'); opt.value=v; dl.appendChild(opt); }); const info=$('#vorschlaege-info'); info.textContent = options.length ? `${options.length} Vorschläge geladen` : 'Keine Vorschläge gefunden'; }
    function uniqSorted(arr){ return [...new Set(arr.filter(Boolean).map(s=>String(s).trim()))].sort((a,b)=>a.localeCompare(b,'de')); }
    function extractFilenames(list){
      const out=[]; (list||[]).forEach(row=>{
        if(typeof row==='string'){ out.push(row); return; }
        if(!row || typeof row!=='object') return;
        [row.filename,row.file_name,row.original_filename,row.name,row.key,row.file,row.source_filename,row.basename]
          .forEach(v=>{ if(typeof v==='string' && v.trim()) out.push(v.trim()); });
        if(row.parsed_json && typeof row.parsed_json.filename==='string'){ out.push(row.parsed_json.filename.trim()); }
      });
      return uniqSorted(out).filter(v=>/\.[a-z0-9]{2,5}$/i.test(v));
    }
    async function loadFilenameSuggestions(){
      const info=$('#vorschlaege-info'); info.textContent='Lade Vorschläge …';
      try{ const r=await fetchWithAuth(endpointBase, {method: "GET", headers:{"Accept-Profile": "x_950_import"}}); if(!r.ok) throw new Error('HTTP '+r.status);
           const list=await r.json(); filenameSuggestions=extractFilenames(list); populateDatalist(filenameSuggestions);
      }catch(err){ info.textContent='Vorschläge konnten nicht geladen werden: '+err.message; toast('Vorschläge konnten nicht geladen werden','err'); }
    }

    async function loadData(){
      const fname = ($('#src_filename').value || '').trim() || defaultSrcFilename; currentSrcFilename=fname;
      $('#loading').style.display=''; const err=$('#error'); err.style.display='none'; err.textContent='';
      try{
        const r = await fetchWithAuth(buildUrl(fname), {method: "GET", headers:{"Accept-Profile": "x_950_import"}}); if(!r.ok) throw new Error("Fehler beim Laden der Daten.");
        const apiData = await r.json();
        if(!Array.isArray(apiData) || !apiData[0]?.parsed_json) throw new Error("Antwortformat unerwartet.");
        const data = apiData[0].parsed_json;
        clearLoadError();
        fillKopf(data.kopf ?? {}); fillAG(data.arbeitsgaenge ?? []); $('#ag-wrap').style.display=''; clearLoadError();
        const dst=$('#dst_filename'); if(!dst.value) dst.value=deriveDstFilename(fname);
        toast('Datei geladen: '+fname,'ok');
      }catch(e){ err.style.display=''; err.textContent="Fehler beim Laden: "+e.message; toast('Laden fehlgeschlagen','err'); }
      finally{ $('#loading').style.display='none'; isDirty=false; }
    }

    // --- Formular Daten ---
    function fillKopf(kopf){
      kopfData = kopf;
      for(const id of ["artikelnummer","bezeichnung","text","spannung","losgroesse","zusatzinfo","benutzung_bis","letzte_aenderung"]){
        const el = document.getElementById(id); if(!el) continue;
        const val = kopf[id]; el.value = (val ?? "");
      }
    }
    function fillAG(ags){
      arbeitsgaengeData = (ags||[]).map(a=>({...a}));
      const tbody = $('#ag-tbody'); tbody.innerHTML="";
      (ags||[]).forEach((ag,idx)=>{
        const tr = document.createElement('tr'); tr.dataset.idx=idx;
        tr.innerHTML = `
          <td><input class="w-80" type="number" name="position" value="${ag.position ?? ""}" data-idx="${idx}" data-key="position"></td>
          <td><input name="beschreibung" value="${ag.beschreibung ? escapeHTML(ag.beschreibung) : ""}" data-idx="${idx}" data-key="beschreibung"></td>
          <td><input class="w-100" type="number" step="0.001" name="te_me" value="${ag.te_me ?? ""}" data-idx="${idx}" data-key="te_me"></td>
          <td><input class="w-100" type="number" step="0.001" name="tr" value="${ag.tr ?? ""}" data-idx="${idx}" data-key="tr"></td>
          <td><input class="w-140" name="fgr" value="${ag.fgr ?? ""}" data-idx="${idx}" data-key="fgr"></td>
          <td><input class="w-100" name="status" value="${ag.status ?? ""}" data-idx="${idx}" data-key="status"></td>
          <td><textarea name="details" data-idx="${idx}" data-key="details">${ag.details ?? ""}</textarea></td>
          <td><input class="w-140" name="letzte_aenderung" value="${ag.letzte_aenderung ?? ""}" data-idx="${idx}" data-key="letzte_aenderung"></td>
        `;
        tbody.appendChild(tr);
      });
      // Delegiertes Event für Inputs/Textareas
      tbody.addEventListener('input', onAgInput, { once:true });
      $$('#ag-tbody textarea').forEach(autoResize);
      $('#ag-count').textContent = `${ags?.length ?? 0} Arbeitsgänge`;
    }
    function onAgInput(e){
      if(!(e.target instanceof HTMLElement)) return;
      const inp = e.target;
      if(!('dataset' in inp)) return;
      const idx = parseInt(inp.dataset.idx,10); const key = inp.dataset.key;
      if(!Number.isFinite(idx) || !key) return;
      arbeitsgaengeData[idx][key] = (inp.type==="number") ? (inp.value==="" ? null : Number(inp.value)) : inp.value;
      isDirty = true;
      // erneutes Delegations-Listener setzen
      $('#ag-tbody').addEventListener('input', onAgInput, { once:true });
      if(inp.tagName==='TEXTAREA') autoResize(inp);
    }

    function escapeHTML(str){ return String(str).replace(/[&<>"']/g, m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;', "'":'&#39;' }[m])); }

    async function saveForm(){
      // Kopf übernehmen
      for(const id of ["artikelnummer","bezeichnung","text","spannung","losgroesse","zusatzinfo","benutzung_bis","letzte_aenderung"]){
        const el = document.getElementById(id); if(el) kopfData[id] = el.value;
      }
      const dst = ($('#dst_filename').value || '').trim() || deriveDstFilename(currentSrcFilename);
      const payload = { filename: dst, parsed_json: { kopf: kopfData, arbeitsgaenge: arbeitsgaengeData } };
      const btn = $('#btn-save'); const res = $('#save-result'); res.textContent="Speichere Daten …";
      try{
        setBusy(btn,true);
        const r = await fetchWithAuth(saveUrl, { method:"POST", headers:{ "Content-Type":"application/json","Prefer":"resolution=merge-duplicates","Content-Profile":"x_950_import" }, body: JSON.stringify(payload) });
        if(!r.ok) throw new Error("Fehler beim Speichern: "+r.status);
        await r.json().catch(()=>({success:true}));
        res.innerHTML = '<span class="success">Daten erfolgreich gespeichert.</span>';
        toast('Gespeichert','ok'); isDirty=false;
        await loadFilenameSuggestions();
      }catch(e){
        res.innerHTML = '<span class="error">Speichern fehlgeschlagen: '+(e.message||e)+'</span>';
        toast('Speichern fehlgeschlagen','err');
      }finally{ setBusy(btn,false); }
    }

    // --- Import in ERP ---
    async function importToERP(){
      const fname = ($('#src_filename').value || '').trim();
      const resEl = $('#import-result'); resEl.className='small';
      if(!fname){ resEl.classList.add('error'); resEl.textContent="Kein Quell-Dateiname ausgewählt."; return; }
      if(isDirty && !confirm('Es gibt ungespeicherte Änderungen. Trotzdem importieren?')) return;
      const btn = $('#btn-import'); setBusy(btn,true); resEl.textContent="Import wird ausgeführt …";
      try{
        const r = await fetchWithAuth(importUrl, { method:"POST", headers:{ "Content-Type":"application/json" }, body: JSON.stringify({ p_filename: fname }) });
        if(!r.ok) throw new Error("HTTP "+r.status);
        const j = await r.json();
        if(j && j.success){
          resEl.classList.add('success');
          resEl.textContent = `Import erfolgreich (Artikel: ${j.ak_nr ?? '-'}, OP: ${j.op_ix ?? '-'}) – Temp-Eintrag gelöscht: ${j.deleted_temp ?? 0}.`;
          toast('Import erfolgreich','ok');
          await loadFilenameSuggestions();
          // Maske leeren, wenn Quelle gelöscht wurde
          if((j.deleted_temp ?? 0) > 0){
            clearForm(); $('#src_filename').value=''; $('#dst_filename').value='';
          }
        } else {
          throw new Error(j && j.error ? j.error : "Unbekannter Fehler");
        }
      }catch(e){
        resEl.classList.add('error'); resEl.textContent = "Import fehlgeschlagen: " + (e.message || e);
        toast('Import fehlgeschlagen','err');
      }finally{ setBusy(btn,false); }
    }

    function clearForm(){
      ["artikelnummer","bezeichnung","text","spannung","losgroesse","zusatzinfo","benutzung_bis","letzte_aenderung"].forEach(id=>{ const el=document.getElementById(id); if(el) el.value=""; });
      const tbody = $('#ag-tbody'); if(tbody) tbody.innerHTML=""; $('#ag-wrap').style.display='none'; $('#ag-count').textContent="";
      kopfData = {}; arbeitsgaengeData = []; isDirty=false;
    }

    // Quick-Filter für Arbeitsgänge
    $('#ag-filter').addEventListener('input', ()=>{
      const q = $('#ag-filter').value.trim().toLowerCase();
      $$('#ag-tbody tr').forEach(tr=>{
        const idx = tr.dataset.idx;
        const ag = arbeitsgaengeData[idx];
        const s = (ag?.beschreibung||'') + ' ' + (ag?.fgr||'');
        tr.style.display = s.toLowerCase().includes(q) ? '' : 'none';
      });
    });
    $('#ag-filter').addEventListener('keydown', e=>{ if(e.key==='Enter'){ e.preventDefault(); $('#ag-filter').value=''; $('#ag-filter').dispatchEvent(new Event('input')); } });

    // Events
    $('#btn-load').addEventListener('click', loadData);
    $('#btn-reload').addEventListener('click', loadData);
    $('#btn-refresh-suggest').addEventListener('click', loadFilenameSuggestions);
    $('#btn-import').addEventListener('click', importToERP);

    $('#src_filename').addEventListener('input', function(){
      const v=this.value; const dst=$('#dst_filename'); if(!dst.dataset.touched) dst.value = deriveDstFilename(v);
    });
    $('#src_filename').addEventListener('keydown', e=>{ if(e.key==='Enter'){ e.preventDefault(); loadData(); }});
    $('#dst_filename').addEventListener('input', function(){ this.dataset.touched="1"; });
    $$('#auftragsForm input, #auftragsForm textarea').forEach(el=> el.addEventListener('input', ()=>{ isDirty=true; }));

    // Init
    window.addEventListener('auth-ready', () => {
      setInitialFilenames();
      loadFilenameSuggestions();
      loadData();
    });
  </script>
</body>
</html>

$html$
$BODY$;


CREATE OR REPLACE FUNCTION api.chk_stv_import()
  RETURNS "text/html"
  LANGUAGE sql
  AS $BODY$
  SELECT $html$

<!DOCTYPE html>
<html lang="de">
<head>
  <meta charset="UTF-8">
  <title>Stücklisten-Import</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <script type="module">
    import { getCookie, setupAuth, fetchWithAuth } from './file?path=auth-header.js';
    setupAuth(); // verwendet in der config im Standard => cookieName: 'jwt', headerName: 'Authorization', prefix: 'Bearer'
    window.fetchWithAuth = fetchWithAuth;
    window.dispatchEvent(new Event('auth-ready'));
  </script>
  <style>
    :root{
      --bg:#f6f7fa; --card:#ffffff; --text:#1d2a3a; --muted:#6b7280;
      --brand:#2a9d8f; --brand-2:#457b9d; --danger:#b22; --ok:#217a2a;
      --border:#e5e7eb; --shadow:0 6px 20px rgba(0,0,0,.06);
      --radius:12px; --radius-sm:8px;
      --pad:2em; --gap:16px; --gap-sm:10px;
    }
    *{box-sizing:border-box}
    html,body{height:100%}
    body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Helvetica,Arial,sans-serif;background:var(--bg);margin:0;padding:0 6vw;color:var(--text)}
    h1{color:#264653;margin:24px 0}
    h2{margin:0 0 12px 0}
    .card{background:var(--card);border-radius:var(--radius);box-shadow:var(--shadow);margin:0 0 24px 0;padding:var(--pad)}
    .grid{display:grid;grid-template-columns:repeat(12,1fr);gap:var(--gap)}
    .grid-2col{display:grid;grid-template-columns:1fr 1fr;gap:var(--gap)}
    @media (max-width: 900px){ body{padding:0 4vw} .grid-2col{grid-template-columns:1fr} }
    table{width:100%;border-collapse:separate;border-spacing:0}
    th,td{padding:10px;border-bottom:1px solid var(--border);vertical-align:top}
    th{background:#f3f6fb;text-align:left;position:sticky;top:0;z-index:1}
    tr:nth-child(even) td{background:#fafbfd}
    tr:hover td{background:#f5fbfa}
    .table-wrap{overflow:auto;border:1px solid var(--border);border-radius:var(--radius)}
    .small{font-size:.9em;color:var(--muted)}
    .muted{color:var(--muted)}
    .error{color:var(--danger);font-weight:600;margin:.5em 0}
    .success{color:var(--ok);font-weight:600}
    .loading{color:var(--muted);margin:.5em 0}

    label{display:block;font-size:.9em;color:#5b6775;margin-bottom:4px}
    input,textarea,select{width:100%;font-size:1em;border-radius:var(--radius-sm);border:1px solid #cfd8e3;padding:8px 10px;background:#fff;transition:border .15s, box-shadow .15s}
    input:focus,textarea:focus,select:focus{outline:none;border-color:var(--brand);box-shadow:0 0 0 3px rgba(42,157,143,.18)}
    textarea{resize:vertical;min-height:2.5em}

    .btn{display:inline-flex;align-items:center;gap:8px;border:none;border-radius:10px;padding:.65em 1.2em;font-weight:600;cursor:pointer}
    .btn:disabled{opacity:.6;cursor:not-allowed}
    .btn-primary{background:var(--brand);color:#fff}
    .btn-secondary{background:var(--brand-2);color:#fff}
    .btn-outline{background:#f8fafc;border:1px solid #cfd8e3;color:#1f2937}
    .btn-ghost{background:transparent;color:#1f2937}
    .chip{display:inline-block;padding:.25em .6em;border-radius:999px;border:1px solid var(--border);background:#fff}

    /* Drag & Drop */
    .dropzone{border:2px dashed #9aa6b2;border-radius:var(--radius);background:#f9fbff;padding:1.5em;text-align:center;transition:.15s;cursor:pointer}
    .dropzone.dragover{border-color:var(--brand);background:#eef9f7}
    .dropzone .hint{color:var(--muted)}
    .dropzone input[type=file]{display:none}

    /* Actions bar sticky bottom */
    .actions-bar{position:sticky;bottom:0;z-index:5;margin-top:20px;background:linear-gradient(180deg, rgba(246,247,250,0) 0%, var(--bg) 35%);padding:12px 0}
    .actions{display:flex;gap:10px;flex-wrap:wrap;align-items:center}

    /* Toasts */
    .toasts{position:fixed;right:20px;bottom:20px;display:flex;flex-direction:column;gap:10px;z-index:9999}
    .toast{padding:.7em 1em;border-radius:10px;box-shadow:var(--shadow);background:#ffffffd9;border:1px solid var(--border)}
    .toast.ok{border-color:#bfe7d8}
    .toast.err{border-color:#f2b8b5}

    .w-80{width:80px} .w-100{width:100px} .w-140{width:140px}
    .nowrap{white-space:nowrap}
    .sr-only{position:absolute!important;height:1px;width:1px;overflow:hidden;clip:rect(1px,1px,1px,1px);white-space:nowrap;border:0;padding:0;margin:-1px}
    .spinner{inline-size:1em;block-size:1em;border-radius:50%;border:.2em solid #cfe3df;border-top-color:var(--brand);animation:spin 1s linear infinite}
    @keyframes spin{to{transform:rotate(360deg)}}
  </style>
</head>
<body>
  <h1>Stücklisten-Import</h1>

  <!-- Upload -->
  <section class="card" aria-labelledby="upload-h">
    <h2 id="upload-h">Stückliste als PDF hochladen &amp; extrahieren</h2>
    <div id="upload-status" class="small" aria-live="polite"></div>
    <div id="dropzone" class="dropzone" tabindex="0" role="button" aria-label="PDF hier ablegen oder klicken, um eine Datei zu wählen">
      <div><strong>Datei hierher ziehen</strong> oder klicken, um eine PDF zu wählen.</div>
      <div class="hint">Erlaubt: *.pdf &nbsp;•&nbsp; Upload an <code>./stv_extract__from_pdf__to_json</code></div>
      <input id="file-input" type="file" accept="application/pdf,.pdf" aria-label="PDF auswählen">
    </div>
    <div class="small muted" id="picked-file" aria-live="polite"></div>
    <div style="margin-top:10px;display:flex;gap:10px;align-items:center">
      <button type="button" id="btn-extract" class="btn btn-outline">
        <span class="sr-only">Extrahieren &amp; speichern</span>Extrahieren &amp; speichern
      </button>
      <span class="small chip" id="upload-hint">Ergebnis wird in <code>/x_950_import/pdf_json_temp</code> gespeichert.</span>
    </div>
    <div id="upload-error" class="error" style="display:none" aria-live="assertive"></div>
    <div id="upload-success" class="success" style="display:none" aria-live="polite"></div>
  </section>

  <!-- Datei wählen -->
  <section class="card" aria-labelledby="datei-h">
    <h2 id="datei-h">Datei wählen</h2>
    <div class="grid-2col">
      <div>
        <label for="src_filename">Dateiname (Quelle)</label>
        <input id="src_filename" name="src_filename" list="filevorschlaege" placeholder="14409-G-Stueckliste.pdf" autocomplete="off">
        <datalist id="filevorschlaege"></datalist>
        <div class="muted small" id="vorschlaege-info" aria-live="polite"></div>
      </div>
      <div>
        <label for="dst_filename">Dateiname (Ziel)</label>
        <input id="dst_filename" name="dst_filename" placeholder="14409-G-Stueckliste.pdf">
      </div>
      <div class="nowrap">
        <button type="button" id="btn-load" class="btn btn-primary">Laden</button>
        <button type="button" id="btn-refresh-suggest" class="btn btn-outline">Vorschläge aktualisieren</button>
      </div>
    </div>
  </section>

  <!-- Formular -->
  <form id="stvForm" autocomplete="off" onsubmit="event.preventDefault();saveForm();" aria-labelledby="form-h">
    <section class="card" aria-labelledby="kopf-h">
      <h2 id="kopf-h">Kopfdaten</h2>
      <div class="grid-2col">
        <div><label for="artikelnummer">Artikelnummer</label><input id="artikelnummer" name="artikelnummer" required></div>
        <div><label for="text">Bezeichnung</label><input id="text" name="text"></div>
        <div><label for="datum">Datum</label><input id="datum" name="datum" type="date"></div>
        <div><label for="letzte_aenderung">Letzte Änderung</label><input id="letzte_aenderung" name="letzte_aenderung" type="date"></div>
        <div><label for="losgroesse">Losgröße</label><input id="losgroesse" name="losgroesse" type="number" min="0" step="1" inputmode="numeric"></div>
        <div><label for="spannung">Spannung</label><input id="spannung" name="spannung"></div>
        <div><label for="gewicht_cu">Cu-Gewicht</label><input id="gewicht_cu" name="gewicht_cu"></div>
      </div>
    </section>

    <section class="card" aria-labelledby="pos-h">
      <div class="grid" style="grid-template-columns:1fr auto;align-items:end">
        <h2 id="pos-h" style="grid-column:1/2;margin:0">Positionen</h2>
        <div class="small" id="pos-count" style="justify-self:end"></div>
        <div style="grid-column:1/-1;margin-top:10px;display:flex;gap:10px;align-items:center">
          <label for="pos-filter" class="small">Schnellfilter</label>
          <input id="pos-filter" placeholder="z. B. Beschreibung oder Artikelnummer" style="max-width:320px">
          <span class="small muted">Enter löscht Filter</span>
        </div>
      </div>

      <div id="loading" class="loading" aria-live="polite">Lade Daten …</div>
      <div id="error" class="error" style="display:none" aria-live="assertive"></div>

      <div class="table-wrap" id="pos-wrap" style="display:none">
        <table id="pos-table">
          <thead>
            <tr>
              <th class="w-80">Pos</th>
              <th class="w-140">Artikelnummer</th>
              <th>Beschreibung</th>
              <th class="w-100">Menge</th>
              <th class="w-100">ME</th>
              <th>Zusatzinfo</th>
            </tr>
          </thead>
          <tbody id="pos-tbody"></tbody>
        </table>
      </div>
    </section>

    <div class="actions-bar">
      <div class="actions">
        <button type="submit" id="btn-save" class="btn btn-primary">Daten speichern</button>
        <button type="button" id="btn-import" class="btn btn-secondary">In ERP importieren</button>
        <button type="button" id="btn-reload" class="btn btn-ghost">Neu laden</button>
        <span id="save-result" class="small" aria-live="polite"></span>
        <span id="import-result" class="small" aria-live="polite"></span>
      </div>
    </div>
  </form>

  <div class="toasts" aria-live="polite" aria-atomic="true"></div>

  <script>
    // --- Endpunkte ---
    const endpointBase     = "/pdf_json_temp";  // GET: ?filename=eq.<name>
    const saveUrl          = "/pdf_json_temp";  // POST: { filename, parsed_json }
    const extractEndpoint  = "./stv_extract__from_pdf__to_json"; // POST: { pdf_data: bytea-hex }
    const importUrl        = "./chk_stv_import_json__from_x_950_import"; // POST: { p_filename }

    // --- State ---
    const defaultSrcFilename = "";
    let currentSrcFilename = defaultSrcFilename;
    let kopfData = {};
    let positionenData = [];
    let filenameSuggestions = [];
    let pickedFile = null;
    let isDirty = false;

    // --- Helpers ---
    const $ = sel => document.querySelector(sel);
    const $$ = sel => Array.from(document.querySelectorAll(sel));
    function toast(msg, type='ok'){ const t = document.createElement('div'); t.className='toast '+(type==='err'?'err':'ok'); t.textContent=msg; const box=$('.toasts'); box.appendChild(t); setTimeout(()=>{ t.style.opacity='0'; setTimeout(()=>t.remove(),300) }, 4000); }
    function setBusy(btn, busy=true){ if(!btn)return; if(busy){ btn.disabled=true; const s=document.createElement('span'); s.className='spinner'; s.setAttribute('aria-hidden','true'); btn.__spinner=s; btn.appendChild(s);} else { btn.disabled=false; if(btn.__spinner){ btn.__spinner.remove(); btn.__spinner=null; } } }
    function autoResize(textarea){ textarea.style.height='auto'; textarea.style.height=(textarea.scrollHeight+2)+'px'; }
    window.addEventListener('beforeunload', e=>{ if(isDirty){ e.preventDefault(); e.returnValue=''; }});
    function clearLoadError() {const err = document.getElementById('error'); if (err) { err.textContent = ''; err.style.display = 'none'; }}

    // --- Upload & Extraktion ---
    const dz = $('#dropzone'); const fileInput = $('#file-input'); const btnExtract = $('#btn-extract');
    dz.addEventListener('click', ()=> fileInput.click());
    dz.addEventListener('keydown', e=>{ if(e.key==='Enter' || e.key===' '){ e.preventDefault(); fileInput.click(); }});
    dz.addEventListener('dragover', e=>{ e.preventDefault(); dz.classList.add('dragover'); });
    dz.addEventListener('dragleave', ()=> dz.classList.remove('dragover'));
    dz.addEventListener('drop', e=>{
      e.preventDefault(); dz.classList.remove('dragover');
      if(e.dataTransfer.files && e.dataTransfer.files[0]) handlePickedFile(e.dataTransfer.files[0]);
    });
    fileInput.addEventListener('change', ()=>{ if(fileInput.files && fileInput.files[0]) handlePickedFile(fileInput.files[0]); });
    function handlePickedFile(f){
      if(!/\.pdf$/i.test(f.name)){ showUploadError("Nur PDF-Dateien sind erlaubt."); return; }
      pickedFile = f; hideUploadMessages(); $('#picked-file').textContent = `Gewählt: ${f.name} (${Math.round(f.size/1024)} KB)`;
    }
    function hideUploadMessages(){ const e=$('#upload-error'), s=$('#upload-success'); e.style.display='none'; e.textContent=''; s.style.display='none'; s.textContent=''; }
    function showUploadError(msg){ const el=$('#upload-error'); el.style.display=''; el.textContent=msg; toast(msg,'err'); }
    function showUploadSuccess(msg){ const el=$('#upload-success'); el.style.display=''; el.textContent=msg; toast(msg,'ok'); }
    function arrayBufferToHex(ab){ const bytes=new Uint8Array(ab); const hex=Array.from(bytes,b=>b.toString(16).padStart(2,'0')).join(''); return '\\x'+hex; }

    btnExtract.addEventListener('click', async ()=>{
      hideUploadMessages(); const status=$('#upload-status'); if(!pickedFile){ showUploadError("Bitte zuerst eine PDF auswählen."); return; }
      try{
        setBusy(btnExtract,true);
        status.textContent="Lese Datei …";
        const buf = await pickedFile.arrayBuffer(); const hex = arrayBufferToHex(buf);
        status.textContent="Sende an Extraktions-Endpunkt …";
        const extractResp = await fetchWithAuth(extractEndpoint,{ method:"POST", headers:{ "Content-Type":"application/json","Prefer":"resolution=merge-duplicates" }, body: JSON.stringify({ pdf_data: hex }) });
        if(!extractResp.ok) throw new Error("Extraktion fehlgeschlagen (HTTP "+extractResp.status+")");
        let extractJson = await extractResp.json().catch(()=>null); if(!extractJson) throw new Error("Leere Antwort vom Extraktions-Endpunkt.");
        let parsed=null;
        if(extractJson.parsed_json) parsed=extractJson.parsed_json;
        else if(Array.isArray(extractJson) && extractJson[0]?.parsed_json) parsed=extractJson[0].parsed_json;
        else if(extractJson.kopf || extractJson.positionen) parsed=extractJson; // Direkte Antwort
        if(!parsed) throw new Error("Antwort enthält kein erwartetes parsed_json.");
        status.textContent="Speichere Ergebnis …";
        const storeFilename = pickedFile.name;
        const saveResp = await fetchWithAuth(saveUrl,{ method:"POST", headers:{ "Content-Type":"application/json","Prefer":"resolution=merge-duplicates","Content-Profile":"x_950_import" }, body: JSON.stringify({ filename: storeFilename, parsed_json: parsed })});
        if(!saveResp.ok) throw new Error("Speichern fehlgeschlagen (HTTP "+saveResp.status+")");
        await loadFilenameSuggestions();
        $('#src_filename').value = storeFilename;
        if(!$('#dst_filename').dataset.touched) $('#dst_filename').value = deriveDstFilename(storeFilename);
        clearLoadError();
        fillKopf(parsed.kopf ?? {}); fillPositionen(parsed.positionen ?? []); $('#pos-wrap').style.display='';
        status.textContent=""; showUploadSuccess("Extraktion und Speicherung erfolgreich.");
      }catch(e){ showUploadError(e.message || String(e)); } finally{ setBusy(btnExtract,false); }
    });

    // --- Vorschläge & Dateiladen ---
    function buildUrl(fname){ return endpointBase + "?filename=eq." + encodeURIComponent(fname); }
    function deriveDstFilename(src){ return (src||""); } // bewusst: Ziel=Quelle
    function setInitialFilenames(){ const s=$('#src_filename'), d=$('#dst_filename'); if(!s.value) s.value=defaultSrcFilename; if(!d.value) d.value=deriveDstFilename(s.value); }
    function populateDatalist(options){ const dl=$('#filevorschlaege'); dl.innerHTML=''; options.forEach(v=>{ const opt=document.createElement('option'); opt.value=v; dl.appendChild(opt); }); const info=$('#vorschlaege-info'); info.textContent = options.length ? `${options.length} Vorschläge geladen` : 'Keine Vorschläge gefunden'; }
    function uniqSorted(arr){ return [...new Set(arr.filter(Boolean).map(s=>String(s).trim()))].sort((a,b)=>a.localeCompare(b,'de')); }
    function extractFilenames(list){
      const out=[]; (list||[]).forEach(row=>{
        if(typeof row==='string'){ out.push(row); return; }
        if(!row || typeof row!=='object') return;
        [row.filename,row.file_name,row.original_filename,row.name,row.key,row.file,row.source_filename,row.basename]
          .forEach(v=>{ if(typeof v==='string' && v.trim()) out.push(v.trim()); });
        if(row.parsed_json && typeof row.parsed_json.filename==='string'){ out.push(row.parsed_json.filename.trim()); }
      });
      return uniqSorted(out).filter(v=>/\.[a-z0-9]{2,5}$/i.test(v));
    }
    async function loadFilenameSuggestions(){
      const info=$('#vorschlaege-info'); info.textContent='Lade Vorschläge …';
      try{ const r=await fetchWithAuth(endpointBase, {method: "GET", headers:{"Accept-Profile": "x_950_import"}}); if(!r.ok) throw new Error('HTTP '+r.status);
           const list=await r.json(); filenameSuggestions=extractFilenames(list); populateDatalist(filenameSuggestions);
      }catch(err){ info.textContent='Vorschläge konnten nicht geladen werden: '+err.message; toast('Vorschläge konnten nicht geladen werden','err'); }
    }

    async function loadData(){
      const fname = ($('#src_filename').value || '').trim() || defaultSrcFilename; currentSrcFilename=fname;
      $('#loading').style.display=''; const err=$('#error'); err.style.display='none'; err.textContent='';
      try{
        const r = await fetchWithAuth(buildUrl(fname), {method: "GET", headers:{"Accept-Profile": "x_950_import"}}); if(!r.ok) throw new Error("Fehler beim Laden der Daten.");
        const apiData = await r.json();
        if(!Array.isArray(apiData) || !apiData[0]?.parsed_json) throw new Error("Antwortformat unerwartet.");
        const data = apiData[0].parsed_json;
        clearLoadError();
        fillKopf(data.kopf ?? {}); fillPositionen(data.positionen ?? []); $('#pos-wrap').style.display=''; clearLoadError();
        const dst=$('#dst_filename'); if(!dst.value) dst.value=deriveDstFilename(fname);
        toast('Datei geladen: '+fname,'ok');
      }catch(e){ err.style.display=''; err.textContent="Fehler beim Laden: "+e.message; toast('Laden fehlgeschlagen','err'); }
      finally{ $('#loading').style.display='none'; isDirty=false; }
    }

    // --- Formular Daten ---
    function fillKopf(kopf){
      kopfData = kopf;
      for(const id of ["artikelnummer","text","datum","letzte_aenderung","losgroesse","spannung","gewicht_cu"]){
        const el = document.getElementById(id); if(!el) continue;
        const val = kopf[id]; el.value = (val ?? "");
      }
    }

    function fillPositionen(pos){
      positionenData = (pos||[]).map(a=>({...a}));
      const tbody = $('#pos-tbody'); tbody.innerHTML="";
      (pos||[]).forEach((p,idx)=>{
        const tr = document.createElement('tr'); tr.dataset.idx=idx;
        tr.innerHTML = `
          <td><input class="w-80" type="number" name="position" value="${p.position ?? ""}" data-idx="${idx}" data-key="position"></td>
          <td><input class="w-140" name="artikelnummer" value="${p.artikelnummer ? escapeHTML(p.artikelnummer) : ""}" data-idx="${idx}" data-key="artikelnummer"></td>
          <td><textarea name="beschreibung" data-idx="${idx}" data-key="beschreibung">${p.beschreibung ?? ""}</textarea></td>
          <td><input class="w-100" type="number" step="0.001" name="menge" value="${p.menge ?? ""}" data-idx="${idx}" data-key="menge"></td>
          <td><input class="w-100" name="mengeneinheit" value="${p.mengeneinheit ?? ""}" data-idx="${idx}" data-key="mengeneinheit"></td>
          <td><textarea name="zusatzinfo" data-idx="${idx}" data-key="zusatzinfo">${p.zusatzinfo ?? ""}</textarea></td>
        `;
        tbody.appendChild(tr);
      });
      // Delegiertes Event für Inputs/Textareas
      tbody.addEventListener('input', onPosInput, { once:true });
      $$('#pos-tbody textarea').forEach(autoResize);
      $('#pos-count').textContent = `${pos?.length ?? 0} Positionen`;
    }

    function onPosInput(e){
      if(!(e.target instanceof HTMLElement)) return;
      const inp = e.target;
      if(!('dataset' in inp)) return;
      const idx = parseInt(inp.dataset.idx,10); const key = inp.dataset.key;
      if(!Number.isFinite(idx) || !key) return;
      positionenData[idx][key] = (inp.type==="number") ? (inp.value==="" ? null : Number(inp.value)) : inp.value;
      isDirty = true;
      // erneutes Delegations-Listener setzen
      $('#pos-tbody').addEventListener('input', onPosInput, { once:true });
      if(inp.tagName==='TEXTAREA') autoResize(inp);
    }

    function escapeHTML(str){ return String(str).replace(/[&<>"']/g, m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;', "'":'&#39;' }[m])); }

    async function saveForm(){
      // Kopf übernehmen
      for(const id of ["artikelnummer","text","datum","letzte_aenderung","losgroesse","spannung","gewicht_cu"]){
        const el = document.getElementById(id); if(el) kopfData[id] = el.value;
      }
      const dst = ($('#dst_filename').value || '').trim() || deriveDstFilename(currentSrcFilename);
      const payload = { filename: dst, parsed_json: { kopf: kopfData, positionen: positionenData } };
      const btn = $('#btn-save'); const res = $('#save-result'); res.textContent="Speichere Daten …";
      try{
        setBusy(btn,true);
        const r = await fetchWithAuth(saveUrl, { method:"POST", headers:{ "Content-Type":"application/json","Prefer":"resolution=merge-duplicates","Content-Profile":"x_950_import" }, body: JSON.stringify(payload) });
        if(!r.ok) throw new Error("Fehler beim Speichern: "+r.status);
        await r.json().catch(()=>({success:true}));
        res.innerHTML = '<span class="success">Daten erfolgreich gespeichert.</span>';
        toast('Gespeichert','ok'); isDirty=false;
        await loadFilenameSuggestions();
      }catch(e){
        res.innerHTML = '<span class="error">Speichern fehlgeschlagen: '+(e.message||e)+'</span>';
        toast('Speichern fehlgeschlagen','err');
      }finally{ setBusy(btn,false); }
    }

    // --- Import in ERP ---
    async function importToERP(){
      const fname = ($('#src_filename').value || '').trim();
      const resEl = $('#import-result'); resEl.className='small';
      if(!fname){ resEl.classList.add('error'); resEl.textContent="Kein Quell-Dateiname ausgewählt."; return; }
      if(isDirty && !confirm('Es gibt ungespeicherte Änderungen. Trotzdem importieren?')) return;
      const btn = $('#btn-import'); setBusy(btn,true); resEl.textContent="Import wird ausgeführt …";
      try{
        const r = await fetchWithAuth(importUrl, { method:"POST", headers:{ "Content-Type":"application/json" }, body: JSON.stringify({ p_filename: fname }) });
        if(!r.ok) throw new Error("HTTP "+r.status);
        const j = await r.json();
        if(j && j.success){
          resEl.classList.add('success');
          // Kurzfassung; Details stehen im JSON (j.inserted.*)
          resEl.textContent = `Import erfolgreich – Temp-Eintrag gelöscht: ${j.deleted_temp ?? 0}.`;
          toast('Import erfolgreich','ok');
          await loadFilenameSuggestions();
          if((j.deleted_temp ?? 0) > 0){
            clearForm(); $('#src_filename').value=''; $('#dst_filename').value='';
          }
        } else {
          throw new Error(j && j.error ? j.error : "Unbekannter Fehler");
        }
      }catch(e){
        resEl.classList.add('error'); resEl.textContent = "Import fehlgeschlagen: " + (e.message || e);
        toast('Import fehlgeschlagen','err');
      }finally{ setBusy(btn,false); }
    }

    function clearForm(){
      ["artikelnummer","text","datum","letzte_aenderung","losgroesse","spannung","gewicht_cu"].forEach(id=>{ const el=document.getElementById(id); if(el) el.value=""; });
      const tbody = $('#pos-tbody'); if(tbody) tbody.innerHTML=""; $('#pos-wrap').style.display='none'; $('#pos-count').textContent="";
      kopfData = {}; positionenData = []; isDirty=false;
    }

    // Quick-Filter für Positionen
    $('#pos-filter').addEventListener('input', ()=>{
      const q = $('#pos-filter').value.trim().toLowerCase();
      $$('#pos-tbody tr').forEach(tr=>{
        const idx = tr.dataset.idx;
        const p = positionenData[idx];
        const s = (p?.beschreibung||'') + ' ' + (p?.artikelnummer||'');
        tr.style.display = s.toLowerCase().includes(q) ? '' : 'none';
      });
    });
    $('#pos-filter').addEventListener('keydown', e=>{ if(e.key==='Enter'){ e.preventDefault(); $('#pos-filter').value=''; $('#pos-filter').dispatchEvent(new Event('input')); } });

    // Events
    $('#btn-load').addEventListener('click', loadData);
    $('#btn-reload').addEventListener('click', loadData);
    $('#btn-refresh-suggest').addEventListener('click', loadFilenameSuggestions);
    $('#btn-import').addEventListener('click', importToERP);

    $('#src_filename').addEventListener('input', function(){
      const v=this.value; const dst=$('#dst_filename'); if(!dst.dataset.touched) dst.value = deriveDstFilename(v);
    });
    $('#src_filename').addEventListener('keydown', e=>{ if(e.key==='Enter'){ e.preventDefault(); loadData(); }});
    $('#dst_filename').addEventListener('input', function(){ this.dataset.touched="1"; });
    $$('#stvForm input, #stvForm textarea').forEach(el=> el.addEventListener('input', ()=>{ isDirty=true; }));

    // Init
    window.addEventListener('auth-ready', () => {
      setInitialFilenames();
      loadFilenameSuggestions();
      loadData();
    });
  </script>
</body>
</html>

$html$;
$BODY$;