CREATE OR REPLACE FUNCTION tabk.abk__search_fuzzy( _needle text )
  RETURNS TABLE( ab_ix integer, name text )
  LANGUAGE plpgsql STABLE 
  AS $$
  DECLARE

      _parts text[];
      
      _regex_excape_characters CONSTANT text := E'\\-[]{}()*+?.,^$|';
      _char char; 

      _where_parts text[] := array[]::text[];
      _field_parts text[] := array[]::text[];

      _fields text[] := array[ 'abk.ab_ap_nr', 'abk.ab_ix', 'art.ak_bez' ];

      _query text :=
          $query$
            WITH
              _search AS (

                  SELECT 
                    abk.ab_ix, 
                    concat( abk.ab_ix, ' » ', ab_ap_nr::text, ' ', ak_bez ) AS description
                  FROM abk
                  LEFT JOIN art ON ak_nr = ab_ap_nr
                  WHERE ab_done IS FALSE 
                    AND %s
              )

              SELECT _search.ab_ix, _search.description
              FROM _search
              ORDER BY min(
                ( _search.ab_ix::text <-> %L )::numeric,
                ( _search.description <-> %L )::numeric
              )
              LIMIT 25
            ;
          $query$;
  BEGIN
  
      IF _needle IS NULL THEN
          RAISE EXCEPTION ' _needle cannot be »NULL«';
      END IF;
  
      -- escape special chars
      FOREACH _char IN ARRAY regexp_split_to_array( _regex_excape_characters, '' ) LOOP
          _needle := replace( _needle, _char, E'\\' || _char );
      END LOOP;
      
      _parts := regexp_split_to_array( _needle, ' ' );

      -- we are splitting up words by spaces and combine them into queries
      -- exmple "fancy search" will be split into two searchterms
      -- 
      -- WHERE ( column_a ~* "fancy" OR column_b ~* "fance" ) 
      --   AND ( column_a ~* "search" OR column_b ~* "search" ) 
      FOR _search_part_index IN 1..cardinality( _parts )
      LOOP

          FOR _search_field_index IN 1..cardinality( _fields )
          LOOP

              -- reset _fields_parts on first iteration
              IF _search_field_index = 1 THEN
                  _field_parts := array[]::text[];
              END IF;

              _field_parts :=
                  _field_parts ||
                  format( '%s ~* %L',
                      _fields[ _search_field_index ],
                      _parts[ _search_part_index ]
                  )
              ;

          END LOOP;

          _where_parts :=
              _where_parts ||
              concat(
                  '( ',
                  array_to_string( _field_parts, ' OR ' ),
                  ' )'
              )
          ;

      END LOOP;

      RETURN QUERY EXECUTE format(
          _query,
          array_to_string( _where_parts, ' AND '),
          _needle,
          _needle
      );

  END $$;
