-- ################################################################################ x2e
-- ################################################################################
-- Entity Component Mapping
-- ################################################################################
CREATE TABLE z_50_customer.code_product_variant_revisions (
    id integer PRIMARY KEY,
    revision_code integer,      --- Revisionscode                --- xtt29794
    revision text,              --- Revision                     --- xtt10412
    label_type integer,         --- Etikettentyp                 --- xtt29795
    short_name text,            --- Kurzwort                     --- xtt29796
    label_type_2 integer,       --- Etikettentyp 2               --- xtt29797
    label_type_3 integer,       --- Etikettentyp 3               --- xtt29798
    label_type_4 integer,       --- Etikettentyp 4               --- xtt29799
    label_type_baumappe integer,--- Baumappe Etikettentyp        --- xtt29800
    template_fsm integer        --- Vorlage FSM                  --- xtt29801
);

CREATE TABLE z_50_customer.code_product_variants (
    id integer PRIMARY KEY,
    variant_code integer,        --- Variantencode        --- xtt29793
    variant text,                --- Variante             --- xtt782
    product_code_id integer
);


CREATE TABLE z_50_customer.code_products (
    id integer PRIMARY KEY,
    product_code integer,         --- Produktcode          --- xtt29791
    product text,                 --- Produkt              --- xtt3510
    production_area_id integer,   --- Produktionsbereich   --- xtt29792
);


CREATE TABLE z_50_customer.entitycomponentmapping (
    entity_component_id INTEGER PRIMARY KEY,
    entity_id VARCHAR(40) NOT NULL REFERENCES art(ak_nr) ON DELETE CASCADE, --Artikelnummer
    product_id integer NOT NULL REFERENCES z_50_customer.code_products(id) ON DELETE CASCADE, --- Produkt-ID                 --- xtt29802
    product_variants integer NOT NULL REFERENCES z_50_customer.code_product_variants(id) ON DELETE CASCADE, --- Produktvariante            --- xtt29803
    product_revisions integer NOT NULL REFERENCES z_50_customer.code_product_variant_revisions(id) ON DELETE CASCADE, --- Produktrevisionen          --- xtt29804
    mount integer
);

-- ################################################################################
-- Parameter für Labelauswahl an Tabelle 'z_50_customer.code_product_variant_revisions'
-- ################################################################################

INSERT INTO recnogroup (reg_bez, reg_gruppe, reg_pname, reg_paramtype, reg_default, reg_exclusive, reg_schema, reg_tablename, reg_autoinsert, reg_pos)
VALUES ('Label', 'Label', 'Label.Typ', 'ptENUM', NULL, true, 'z_50_customer', 'code_product_variant_revisions', false, 1) ON CONFLICT DO NOTHING;

INSERT INTO recnoenums  (rege_reg_pname, rege_code, rege_bez, rege_pos)
VALUES('Label.Typ', '1', 'Kein Etikett drucken' , 1),
      ('Label.Typ', '2', '60x40 Typenschild Xoraya' , 2),
      ('Label.Typ', '3', '25x8 Komponentenetikett' , 3),
      ('Label.Typ', '4', '60x40 Leihgabe (engl.)' , 4),
      ('Label.Typ', '5', '60x40 Typenschild CarPC' , 5),
      ('Label.Typ', '6', '60x40 Typenschild ZGW' , 6),
      ('Label.Typ', '7', '15x6,5 Komponentenetikett' , 7),
      ('Label.Typ', '8', '60x40 Typenschild Xoraya 1810' , 8),
      ('Label.Typ', '9', '60x40 Typenschild Daimler CGW' , 9),
      ('Label.Typ', '10', '60x40 Typenschild Xoraya V5' , 10),
      ('Label.Typ', '11', '60x40 Auslieferungsetikett Datenlogger' , 11),
      ('Label.Typ', '12', '60x40 Auslieferungsetikett Connect' , 12),
      ('Label.Typ', '13', '60x40 Typenschild Connect' , 13),
      ('Label.Typ', '16', '25x8 Fertigung für extern' , 16),
      ('Label.Typ', '20', '60x40 Typenschild Minilogger V5' , 20),
      ('Label.Typ', '21', '60x40 Auslieferungsetikette Minilogger V5' , 21),
      ('Label.Typ', '22', '60x40 Typenschild ML-Z7' , 22),
      ('Label.Typ', '23', '60x40 Auslieferungsetikette ML-Z7' , 23),
      ('Label.Typ', '25', '60x40 Typenschild LIN-Breakoutbox' , 25),
      ('Label.Typ', '31', '30x20 Typenschild klein Levelshifter CE' , 31),
      ('Label.Typ', '32', '60x40 Typenschild UPoB' , 32),
      ('Label.Typ', '33', '30x20 Typenschild klein USB-Relais' , 33),
      ('Label.Typ', '34', '30x20 Typenschild klein AEC CE' , 34),
      ('Label.Typ', '35', '30x20 Typenschild klein OABR2CAN CE' , 35),
      ('Label.Typ', '36', '25x8 Kabel' , 36),
      ('Label.Typ', '37', '30x20 Kabel' , 37),
      ('Label.Typ', '39', '60x40 Typenschild Ext. Storage' , 39),
      ('Label.Typ', '40', '10x6,5 Komponentenetikett' , 40),
      ('Label.Typ', '41', '60x40 Typenschild Datacube' , 41),
      ('Label.Typ', '42', '60x40 Auslieferungsetikette Datacube' , 42),
      ('Label.Typ', '43', '60x40 Typenschild Xoraya-Z7' , 43),
      ('Label.Typ', '44', '60x40 Auslieferungsetikette Xoraya-Z7' , 44),
      ('Label.Typ', '45', '30x20 Typenschild M2 ext. Storage' , 45),
      ('Label.Typ', '46', '60x40 Typenschild Daimler Adapterbox' , 46),
      ('Label.Typ', '47', '60x40 Typenschild Micrologger-Z7' , 47),
      ('Label.Typ', '48', '60x40 Auslieferungsetikette Xoraya-Micrologger-Z7' , 48),
      ('Label.Typ', '49', '60x40 Typenschild Microtastkopf-Z7' , 49),
      ('Label.Typ', '50', '60x40 Auslieferungsetikette Xoraya-Microtastkopf-Z7' , 50),
      ('Label.Typ', '52', '60x40 Auslieferungsetikett  Xoraya N8000' , 52),
      ('Label.Typ', '53', '25x8 Komponentenetikett 2 (Kühlkörper)' , 53),
      ('Label.Typ', '54', '60x40 Auslieferungsetikett Loggeraufbau Daimler' , 54),
      ('Label.Typ', '55', '60x40 Typenschild N8000 Massenspeicher' , 55),
      ('Label.Typ', '56', '60x40 Auslieferungsetikett N8000 Massenspeicher' , 56),
      ('Label.Typ', '57', '60x40 Typenschild Datacube N3' , 57),
      ('Label.Typ', '58', '60x40 Auslieferungsetikette Datacube N3' , 58),
      ('Label.Typ', '59', '60x40 Typenschild Datacube N1' , 59),
      ('Label.Typ', '60', '60x40 Auslieferungsetikett Datacube N1' , 60),
      ('Label.Typ', '61', '60x40 Typenschild Xoraya N16000' , 61),
      ('Label.Typ', '62', '60x40 Auslieferungsetikett  Xoraya N16000' , 62),
      ('Label.Typ', '63', '60x40 Typenschild ML-N4000' , 63),
      ('Label.Typ', '64', '60x40 Auslieferungsetikett ML-N4000' , 64),
      ('Label.Typ', '65', '60x40 Typenschild Xoraya N4000' , 65),
      ('Label.Typ', '66', '60x40 Auslieferungsetikett Xoraya N4000' , 66),
      ('Label.Typ', '67', '25x8 Komponentenetikett 2 (Baumappe)' , 67),
      ('Label.Typ', '68', '60x40 Typenschild Xoraya N3000' , 68),
      ('Label.Typ', '69', '25x8 Komponentenetikett (Artikel)' , 69),
      ('Label.Typ', '70', '25x8 Komponentenetikett (Artikel+Charge)' , 70),
      ('Label.Typ', '71', '25x8 Komponentenetikett (Lagerort)' , 71),
      ('Label.Typ', '72', '25x8 Komponentenetikett (Fertigungsauftrag)' , 72),
      ('Label.Typ', '77', '60x40 Arbeitsplatz (weiß)' , 77),
      ('Label.Typ', '78', '30x20 Typenschild klein OABR-to-Ethernet' , 78),
      ('Label.Typ', '79', '25x8 Komponentenetikett (Menge)' , 79),
      ('Label.Typ', '80', '60x40 Typenschild Connect-ZU' , 80) ON CONFLICT DO NOTHING;

-- ################################################################################
-- Parameter am Artikel für Etiketten
-- ################################################################################
INSERT INTO recnogroup (reg_bez, reg_gruppe, reg_pname, reg_paramtype, reg_default, reg_exclusive, reg_tablename, reg_autoinsert, reg_pos) VALUES
('NOMINAL-VOLTAGE', 'Etiketten', 'etik.nomvolt', 'ptVARCHAR', NULL, true, 'art', false, 1),
('POWER-CONSUMPTION', 'Etiketten', 'etik.pwrcon', 'ptVARCHAR', NULL, true, 'art', false, 2),
('TYPE-NAME', 'Etiketten', 'etik.typ', 'ptVARCHAR', NULL, true, 'art', false, 3),
('RAM-SIZE / DRIVE-SIZE', 'Etiketten', 'etik.ramsize', 'ptVARCHAR', NULL, true, 'art', false, 4);

-- ################################################################################
-- SN Generator
-- ################################################################################
CREATE OR REPLACE FUNCTION z_50_customer.x2e__esn__is( _esn varchar ) RETURNS boolean AS $$

  -- handelt es sich hier um eine erweiterte Seriennumer im x2e-Format?
  --   Format hhhh_hhhh_hhhh_hhhhhh wobei 'h' hier eine Hexadezimalziffer repräsentiert
  --   inhaltlich: <product>_<variant>_<revision>_<lfd_nr>
  --   Produkt, Variante und Revision ergeben gemeinsam die Produktgruppe
  SELECT _esn ~ '^([0-9,A-F]{4}[_]){3}[0-9,A-F]{6}$';

$$ LANGUAGE sql IMMUTABLE;
--


CREATE OR REPLACE FUNCTION z_50_customer.x2e__esn__seriennummer__extract( _esn varchar )
RETURNS integer AS $$
DECLARE
  _hex_string varchar;
  _result integer;
BEGIN

  -- wandelt den letzten Teil einer ESN (die laufende Nummer) in eine Dezimalzahl um
  _result := null;
  _hex_string := r FROM right( _esn, 6 ) AS r WHERE z_50_customer.x2e__esn__is( _esn );

  IF _hex_string IS NOT null THEN
    EXECUTE 'SELECT x''' || _hex_string || '''::integer' INTO _result;
  END IF;

  RETURN _result;

END $$ LANGUAGE plpgsql IMMUTABLE;
--


CREATE OR REPLACE FUNCTION z_50_customer.x2e__lagsernr__presetting__create( _ak_nr varchar, _count integer )
RETURNS SETOF varchar AS $$
DECLARE
  _cnt             integer;
  _product_code    integer;
  _variant_code    integer;
  _revision_code   integer;
  _entity_id       varchar;
  _product_variant varchar;
  _max_esn         varchar;
  _sn_dezimal      integer;
  _r               record;
BEGIN

  -- sucht zur Produktgruppe des Artikels die nächsten freien erweiterten Seriennummern
  --   ist die letzte ESN der Produktgruppe des Artikels = <product>_<variant>_<revision>_<lfd_nr>
  --   dann liefert diese Funktion <product>_<variant>_<revision>_<lfd_nr + 1>, alles im Hex-Format

  _cnt := max( _count, 0 );

  -- nur für x2e interessant
  IF _cnt > 0 AND isnumeric( _entity_id ) THEN

    -- das betrifft nur Artikel mit einer im wesentlichen numerischen Artikelnummer
    _entity_id = right( _ak_nr, 8);

    -- bestimmt Produkt, Variante und Revision des Artikels, also seine Produktgruppe
    SELECT product_code,  variant_code,  revision_code
    INTO  _product_code, _variant_code, _revision_code
      FROM art
      JOIN z_50_customer.entitycomponentmapping               ON entity_id = _entity_id::integer
      JOIN z_50_customer.code_products                  AS pr ON product_id = pr.id
      JOIN z_50_customer.code_product_variants          AS vr ON product_variants = vr.id
      JOIN z_50_customer.code_product_variant_revisions AS re ON product_revisions = re.id
      WHERE ak_nr = _ak_nr
      LIMIT 1;

    -- Keine Produktgruppe gefunden? Dann gibt es keine ESN.
    IF _product_code || _variant_code || _revision_code IS NOT null THEN
      -- die ersten drei Komponenten der ESN zusammenbauen
      _product_variant := upper(
          lpad( to_hex( _product_code  ), 4, 0 ) || '_' ||
          lpad( to_hex( _variant_code  ), 4, 0 ) || '_' ||
          lpad( to_hex( _revision_code ), 4, 0 ) || '_'
       );

      -- hole die letzte größte Seriennummer der Produktgruppe
      _max_esn :=
            max ( lgs_sernr )
            FROM lagsernr
            WHERE
                    z_50_customer.x2e__esn__is( lgs_sernr )
                AND position( _product_variant IN lgs_sernr ) = 1;

      -- ermittelt die letzte laufende Seriennummer
      _sn_dezimal := z_50_customer.x2e__esn__seriennummer__extract( _max_esn );
      IF _sn_dezimal IS null THEN
        _sn_dezimal := 0;
      END IF;

      -- gibt die entsprechende Anzahl neu generierten Seriennummern zurück
      FOR _r IN (
        SELECT ( upper( _product_variant || lpad( to_hex( lfd_nr ), 6, 0 ))) :: varchar
        FROM generate_series( _sn_dezimal + 1, _sn_dezimal + _cnt ) AS lfd_nr
      ) LOOP
        RETURN NEXT _r;
      END LOOP;

    END IF;
  END IF;

END $$ LANGUAGE plpgsql STABLE STRICT;
--


CREATE OR REPLACE FUNCTION z_50_customer.x2e__lagsernr__presetting__create( _ld_id integer, _count integer, _w_wen integer )
RETURNS SETOF varchar AS $$
DECLARE
  _ld_aknr  varchar;
  _ld_stk   integer;
  _w_aknr   varchar;
  _w_zugang integer;
  _ak_nr    varchar;
  _cnt      integer;
  _r        record;
BEGIN

  -- Seriennummerngenerator für x2e
  -- ermittelt zunächst Artikel und Anzahl der benötigten Seriennummern
  --   und generiert diese dann

  SELECT ld_aknr, ld_stk INTO _ld_aknr, _ld_stk FROM ldsdok WHERE ld_id = _ld_id;
  SELECT w_aknr, w_zugang INTO _w_aknr, _w_zugang FROM wendat WHERE w_wen = _w_wen;
  _ak_nr := coalesce( _ld_aknr, _w_aknr );
  _cnt   := coalesce( _count, _w_zugang, _ld_stk );

  FOR _r IN ( SELECT z_50_customer.x2e__lagsernr__presetting__create( _ak_nr, _cnt )) LOOP
    RETURN NEXT _r;
  END LOOP;

END $$ LANGUAGE plpgsql STABLE;
--


CREATE OR REPLACE FUNCTION z_50_customer.x2e__lgs_sernr__ak_nr__get( _lgs_sernr varchar ) RETURNS varchar AS $$
  DECLARE
    _product_code    integer;
    _variant_code    integer;
    _revision_code   integer;
    _entity_id       varchar;
  BEGIN

    -- liefert die Artikelnummer zu einer x2e-Seriennummer
    -- Format dieser Seriennummern: pppp_vvvv_rrrr_ssssss
      -- pppp - Produktcode als Hexzahl
      -- vvvv - Produktvariante als Hexzahl
      -- rrrr - Revision als Hexzahl
      -- ssssss - laufende Seriennummer als Hexzahl
    -- Aus der Kombination aus Produktcode, Produktvariante und Revision
    -- ergibt sich die Artikelnummer eindeutig.

    _product_code  := hex_to_integer( substring( _lgs_sernr FROM  1 FOR 4));
    _variant_code  := hex_to_integer( substring( _lgs_sernr FROM  6 FOR 4));
    _revision_code := hex_to_integer( substring( _lgs_sernr FROM 11 FOR 4));

    SELECT entity_id INTO _entity_id
    FROM z_50_customer.entitycomponentmapping
    JOIN z_50_customer.code_products                  AS pr ON product_id = pr.id AND product_code = _product_code
    JOIN z_50_customer.code_product_variants          AS vr ON product_variants = vr.id AND variant_code = _variant_code
    JOIN z_50_customer.code_product_variant_revisions AS re ON product_revisions = re.id AND revision_code = _revision_code
    LIMIT 1;

    -- Artikelnummer = Entity-ID
    RETURN _entity_id;

  END $$ LANGUAGE plpgsql STRICT STABLE;


CREATE OR REPLACE FUNCTION z_50_customer.x2e__lgs_sernr__matrix( _lgs_sernr varchar ) RETURNS varchar AS $$
  DECLARE
    _ak_nr varchar;
    _labeltypString varchar;
    _labeltyp integer;
    _kabellabel boolean;
    _config varchar;
    _sn varchar;
    _product_id varchar;
    _product_variants varchar;
    _kabel_hex varchar;
    _matrix_code varchar;
    _dmf varchar;
  BEGIN

    -- bestimmt zu einer x2e-Seriennummer einen Matrixcode, welcher spezifische Informationen für den Etikettendruck enthält
    -- Dies erfolgt über die recnokeywords, in welcher einem Artikel ein Labeltyp zugeordnet wird.
    -- Es wird zwischen drei Labeltypen unterschieden:
      -- Kein Label
      -- Kabellabel: Format ppppvvvvssssssss in besa64-Code
      -- Typenschild: Format * ssssss | pppp_vvvv_rrrr | MM.YYYY | --.---- *
      -- Legende
        -- pppp - Produktcode als Hexzahl
        -- vvvv - Produktvariante als Hexzahl
        -- rrrr - Revision als Hexzahl
        -- ssssss - laufende Seriennummer als Hexzahl
        -- MM.YYYY - DMF, "date of manufactoring" = Insertdatum der Seriennummer

    _ak_nr := z_50_customer.x2e__lgs_sernr__ak_nr__get( _lgs_sernr );
    _labeltypString := r_value
      FROM recnokeyword
      JOIN art ON ak_nr = _ak_nr
      WHERE r_dbrid = art.dbrid AND r_reg_pname = 'Label.Typ';

    IF _labeltypString IS null OR NOT isNumeric( _labeltypString ) THEN
      RETURN null;
    END IF;
    _labeltyp := _labeltypString :: numeric;

    -- Labeltyp 1 = Kein Label
    IF _labeltyp = 1 THEN
      RETURN null;
    END IF;

    -- Labeltypen, die als Kabellabel definiert wurden
    IF _labeltyp IN ( 3, 7, 16, 31, 33, 34, 35, 36, 37, 40, 45, 53, 67, 69, 70, 71, 72, 78, 79 ) THEN
      _kabellabel := true;
    ELSE
      _kabellabel := false;
    END IF;

    _config := substring( _lgs_sernr FROM   1 FOR 14);
    _sn     := substring( _lgs_sernr FROM  16 FOR  6);

    IF _kabellabel THEN
      SELECT lpad( to_hex( product_id ), 2, '0' ), lpad( to_hex( product_variants ), 2, '0' )
        INTO _product_id, _product_variants
        FROM z_50_customer.entitycomponentmapping
        WHERE entity_id = _ak_nr
        LIMIT 1;
      _kabel_hex := _product_id || _product_variants || '00' || _sn;
      _matrix_code := hex_to_base64( _kabel_hex );
    ELSE
      _dmf := to_char( COALESCE(insert_date, today()), 'MM.YYYY' ) FROM lagsernr WHERE lgs_sernr = _lgs_sernr;
      _matrix_code := '* ' || _sn || ' | ' || _config || ' | ' || _dmf || ' | --.---- *';
    END IF;

    RETURN _matrix_code;

  END $$ LANGUAGE plpgsql STRICT STABLE;

CREATE OR REPLACE FUNCTION z_50_customer.x2e__lgs_sernr__matrix_code( _lgs_sernr varchar ) RETURNS varchar AS $$
DECLARE
    _config varchar;
    _sn varchar;
    _matrix_code varchar;
    _dmf varchar;
  BEGIN

    -- bestimmt zu einer x2e-Seriennummer einen Matrixcode, welcher spezifische Informationen für den Etikettendruck enthält
      -- Typenschild: Format * ssssss | pppp_vvvv_rrrr | MM.YYYY | --.---- *
      -- Legende
        -- pppp - Produktcode als Hexzahl
        -- vvvv - Produktvariante als Hexzahl
        -- rrrr - Revision als Hexzahl
        -- ssssss - laufende Seriennummer als Hexzahl
        -- MM.YYYY - DMF, "date of manufacturing" = Insertdatum der Seriennummer

    _config := substring( _lgs_sernr FROM   1 FOR 14);
    _sn     := substring( _lgs_sernr FROM  16 FOR  6);
    _dmf := to_char( COALESCE(insert_date, today()), 'MM.YYYY' ) FROM lagsernr WHERE lgs_sernr = _lgs_sernr;
    _matrix_code := '* ' || _sn || ' | ' || _config || ' | ' || _dmf || ' | --.---- *';

    RETURN _matrix_code;

  END $$ LANGUAGE plpgsql STRICT STABLE;

CREATE OR REPLACE FUNCTION z_50_customer.x2e__lgs_sernr__matrix_codeshort( _lgs_sernr varchar ) RETURNS varchar AS $$
DECLARE
    _ak_nr varchar;
    _config varchar;
    _sn varchar;
    _product_id varchar;
    _product_variants varchar;
    _kabel_hex varchar;
    _matrix_code varchar;
  BEGIN

    -- bestimmt zu einer x2e-Seriennummer einen Matrixcode, welcher spezifische Informationen für den Etikettendruck enthält
    -- Dies erfolgt über die recnokeywords, in welcher einem Artikel ein Labeltyp zugeordnet wird.
      -- Kabellabel: Format ppppvvvvssssssss in besa64-Code
      -- Legende
        -- pppp - Produktcode als Hexzahl
        -- vvvv - Produktvariante als Hexzahl
        -- ssssss - laufende Seriennummer als Hexzahl, wird noch mit zwei führende Nullen ergänzt

    _ak_nr := z_50_customer.x2e__lgs_sernr__ak_nr__get( _lgs_sernr );

    _config := substring( _lgs_sernr FROM   1 FOR 14);
    _sn     := substring( _lgs_sernr FROM  16 FOR  6);

    SELECT lpad( to_hex( product_id ), 2, '0' ), lpad( to_hex( product_variants ), 2, '0' )
          INTO _product_id, _product_variants
          FROM z_50_customer.entitycomponentmapping
          WHERE entity_id = _ak_nr
          LIMIT 1;
    _kabel_hex := _product_id || _product_variants || '00' || _sn;
    _matrix_code := hex_to_base64( _kabel_hex );

  RETURN _matrix_code;

  END $$ LANGUAGE plpgsql STRICT STABLE;


-- ################################################################################
-- Schnittstelle Altium Designer (#16097)
-- ################################################################################

-- Erstellt View entity_types.
-- Diese Funktion wird von Funktion TSystem.views__customer__drop() bei x2e gerufen.
CREATE OR REPLACE FUNCTION z_50_customer.entity_types__recreate() RETURNS VOID AS $$
    BEGIN
        CREATE OR REPLACE VIEW z_50_customer.entity_types AS

            WITH ak_zolp1bs AS (
              SELECT DISTINCT ak_zolp1b
              FROM art
              JOIN recnokeyword ON art.dbrid = r_dbrid
              JOIN recnogroup ON reg_pname = r_reg_pname
              WHERE
                        ak_zolp1b IS NOT NULL
                    AND COALESCE( ak_auslauf, 'infinity'::timestamp ) >= date_trunc( 'day', currenttime() )
                    AND NOT COALESCE( r_descr, reg_bez ) = 'keywordsearch'
            )

            SELECT
              ROW_NUMBER () OVER (ORDER BY ak_zolp1b)::integer AS entity_type_no,
              ak_zolp1b AS entity_type_name
            FROM ak_zolp1bs;

    END $$ LANGUAGE plpgsql;
--

-- Löscht View entity_types.
-- Diese Funktion wird von Funktion TSystem.views__customer__create() bei x2e gerufen.
CREATE OR REPLACE FUNCTION z_50_customer.entity_types__drop() RETURNS VOID AS $$
    BEGIN
        DROP VIEW IF EXISTS z_50_customer.entity_types;
    END $$ LANGUAGE plpgsql;
--

-- Erstellet einen View für eine gegebene Entity-Nummer.
CREATE OR REPLACE FUNCTION z_50_customer.create_altium_view( IN_entity_type_no integer ) RETURNS void AS $$
    DECLARE
        _entity_type_name varchar;
        _view_name varchar;
        _column_names text;
        _source_sql text;
        _category_sql text;
        _query text;
    BEGIN
        -- Die Entity-Nummer in den Namen übersetzen.
        SELECT entity_type_name
        INTO _entity_type_name
        FROM z_50_customer.entity_types
        WHERE entity_type_no = IN_entity_type_no;

        -- Den Namen des zu erzeugenden Views bestimmen
        _view_name := 'altium.' || _entity_type_name || 'view';

        -- Die Parameternamen als Liste der Spaltennamen für das SQL-Create-View-Kommando vorbereiten.
        -- Dabei für die Spalten "ak_nr", "ak_bez" und "ak_zolp1b" die Aliase "x2e_number", "entity_name" und "entity_type_name" einsetzen.
        -- Je nach Parameternamen werden noch weitere Aiase eingesetzt.
        WITH _columns AS (
            SELECT 'x2e_number' AS col, 1 AS rnk, '' AS reg_bez
            UNION
            SELECT 'entity_name' AS col, 2 AS rnk, '' AS reg_bez
            UNION
            SELECT 'entity_type_name' AS col, 3 AS rnk, '' AS reg_bez
            UNION
            SELECT DISTINCT CASE reg_bez
                                WHEN 'footprint_ref'
                                    THEN 'Footprint Ref'
                                WHEN 'library_ref'
                                    THEN 'Library Ref'
                                ELSE reg_bez
                              END AS col
                    , 4 AS rnk
                    , reg_bez
            FROM art
            JOIN recnokeyword ON art.dbrid = r_dbrid
            JOIN recnogroup ON reg_pname = r_reg_pname
            WHERE
                      Ak_Zolp1b = _entity_type_name
                  AND COALESCE( ak_auslauf, 'infinity'::timestamp ) >= date_trunc( 'day', currenttime() )
                  AND NOT COALESCE( r_descr, reg_bez ) = 'keywordsearch'
            ORDER BY rnk, reg_bez
        )
        SELECT coalesce( string_agg( E'\n                    "' || col || '" varchar', ',' ), '' ) || E'\n                ' AS column_names
        INTO _column_names
        FROM _columns;

        -- Das innere SQL-Statement für der crosstab-Funktion vorbereiten.
        _source_sql := $s1$
                        SELECT
                                ak_nr
                              , ak_bez
                              , ak_zolp1b
                              , reg_bez AS cat
                              , r_value AS value
                          FROM art
                          JOIN recnokeyword ON r_dbrid = art.dbrid
                          JOIN recnogroup ON reg_pname = r_reg_pname
                          WHERE
                                    Ak_Zolp1b = ''$s1$ || _entity_type_name || $s2$''
                                AND COALESCE( ak_auslauf, ''infinity''::timestamp ) >= date_trunc( ''day'', currenttime() )
                                AND NOT COALESCE( r_descr, reg_bez ) = ''keywordsearch''
                          ORDER BY 1
                      $s2$;

        _category_sql := $c1$
                          SELECT DISTINCT reg_bez as cat
                            FROM art
                            JOIN recnokeyword ON art.dbrid = r_dbrid
                            JOIN recnogroup ON reg_pname = r_reg_pname
                            WHERE
                                      Ak_Zolp1b = ''$c1$ || _entity_type_name || $c2$''
                                  AND COALESCE( ak_auslauf, ''infinity''::timestamp ) >= date_trunc( ''day'', currenttime() )
                                  AND NOT COALESCE( r_descr, reg_bez ) = ''keywordsearch''
                            ORDER BY 1
                        $c2$;

        -- Das SQL-Kommando zum erzeugen des Views zusammenbauen.
        _query := $q1$
            CREATE OR REPLACE VIEW $q1$ || _view_name || $q2$ AS
            SELECT *
              FROM crosstab(
                      '$q2$ || _source_sql || $q3$',
                      '$q3$ || _category_sql || $q4$'
                    ) AS ct ( $q4$ || _column_names || $q5$ )
              ORDER BY 1;
                $q5$;

        RAISE NOTICE E'%: %', _entity_type_name, _query;

        EXECUTE _query;
    END $$ LANGUAGE plpgsql;
--

-- Durchläut alle Entity-Typen und erstellet je einen View.
-- Diese Funktion wird von Funktion TSystem.views__customer__create() bei x2e gerufen.
CREATE OR REPLACE FUNCTION z_50_customer.create_altium_views() RETURNS void AS $$
    BEGIN
        -- Altium-Schema einschließlich der Views löschen.
        -- Das folgende Drop-Statement wird von Funktion TSystem.views__customer__drop() bei x2e gerufen.
        DROP SCHEMA IF EXISTS altium CASCADE;

        -- Altium-Schema wieder erstellen.
        CREATE SCHEMA IF NOT EXISTS altium;

        -- Die Entity-Typen durchlaufen und je einen View erstellen.
        PERFORM z_50_customer.create_altium_view(entity_type_no)
        FROM z_50_customer.entity_types;
    END $$ LANGUAGE plpgsql;
--

--- Ende Schnittstelle Altium Designer (#16097)


-- ################################################################################
-- eGroupware-Schnittstelle (#16795, #20362)
-- ################################################################################

-- Parameter uidnumber in Tabelle recnogroup eintragen.
INSERT INTO recnogroup( reg_tablename , reg_gruppe, reg_bez               , reg_pname   , reg_paramtype )
SELECT                  'llv'         , 'System'  , 'Windows Benutzer-ID' , 'uidnumber' , 'ptINTEGER'
WHERE NOT EXISTS ( SELECT TRUE FROM recnogroup WHERE reg_pname = 'uidnumber' );
--

-- Settings für die eGroupware-Kategorien anlegen.
-- Diese werden in Funktion z_50_customer.egw_cal__export() verwendet.
DO $$
DECLARE
    _iconnname text := 'egconn'; -- eigener (interner) Verbindungsname zur Egroupware-DB.
    _now      bigint;
    _sql      varchar;
    _row_cnt  integer;
    _cat_id   integer;

BEGIN
  IF tsystem.settings__gettext( 'X2E_EGroupWare_DBLink' ) > '' THEN
      -- Eine eigene Verbindung zu Egroupware-DB aufbauen.
      PERFORM dblink_connect_u( _iconnname, tsystem.settings__gettext( 'X2E_EGroupWare_DBLink' ) );

      -- Jetzt
      _now := EXTRACT(
                EPOCH FROM (
                  now()
                )::timestamp at time zone 'Europe/Berlin'
              )::bigint;

      -- Urlaub (lila)
      _sql := format( 'SELECT count( * ) FROM egw_categories WHERE cat_appname = %L AND cat_access = %L AND cat_name = %L', 'calendar', 'public', 'Urlaub' );
      SELECT row_cnt INTO _row_cnt FROM dblink( _iconnname, _sql ) AS ( row_cnt integer );
      -- Datensatz gibt es noch nicht.
      IF _row_cnt = 0 THEN
          --> Insert
          _sql := format( 'INSERT INTO egw_categories( cat_access, cat_appname, cat_name          , cat_description          , cat_data                       , last_mod ) ' ||
                          'VALUES                    ( %L        , %L         , %L                , %L                       , %L                             , %L       ) ' ||
                          'RETURNING cat_id;',
                                                        'public'  , 'calendar' , 'Urlaub'          , 'Urlaub'                 , '{"color":"#7f00ff","icon":""}', _now             -- INSERT
          );
          SELECT cat_id INTO _cat_id FROM dblink( _iconnname, _sql ) AS ( cat_id integer );
          PERFORM tsystem.settings__set( 'X2E_EGroupWare_Kategorie_Urlaub', _cat_id::varchar(64) );
      -- Datensatz existiert schon.
      ELSE
          --> Set Setting
          _sql := format( 'SELECT min( cat_id ) FROM egw_categories WHERE cat_appname = %L AND cat_access = %L AND cat_name = %L', 'calendar', 'public', 'Urlaub' );
          PERFORM tsystem.settings__set( 'X2E_EGroupWare_Kategorie_Urlaub', cat_id )
          FROM dblink( _iconnname, _sql ) AS ( cat_id varchar(64) );
      END IF;

      -- Urlaub 1/2 (lila)
      _sql := format( 'SELECT count( * ) FROM egw_categories WHERE cat_appname = %L AND cat_access = %L AND cat_name = %L', 'calendar', 'public', 'Urlaub 1/2' );
      SELECT row_cnt INTO _row_cnt FROM dblink( _iconnname, _sql ) AS ( row_cnt integer );
      -- Datensatz gibt es noch nicht.
      IF _row_cnt = 0 THEN
          --> Insert
          _sql := format( 'INSERT INTO egw_categories( cat_access, cat_appname, cat_name          , cat_description          , cat_data                       , last_mod ) ' ||
                          'VALUES                    ( %L        , %L         , %L                , %L                       , %L                             , %L       ) ' ||
                          'RETURNING cat_id;',
                                                        'public'  , 'calendar' , 'Urlaub 1/2'      , 'Urlaub 1/2'             , '{"color":"#7f00ff","icon":""}', _now             -- INSERT
          );
          SELECT cat_id INTO _cat_id FROM dblink( _iconnname, _sql ) AS ( cat_id integer );
          PERFORM tsystem.settings__set( 'X2E_EGroupWare_Kategorie_Urlaub_einhalb', _cat_id::varchar(64) );
      -- Datensatz existiert schon.
      ELSE
          --> Set Setting
          _sql := format( 'SELECT min( cat_id ) FROM egw_categories WHERE cat_appname = %L AND cat_access = %L AND cat_name = %L', 'calendar', 'public', 'Urlaub 1/2' );
          PERFORM tsystem.settings__set( 'X2E_EGroupWare_Kategorie_Urlaub_einhalb', cat_id )
          FROM dblink( _iconnname, _sql ) AS ( cat_id varchar(64) );
      END IF;

      -- Stundenausgleich (lila)
      _sql := format( 'SELECT count( * ) FROM egw_categories WHERE cat_appname = %L AND cat_access = %L AND cat_name = %L', 'calendar', 'public', 'Stundenausgleich' );
      SELECT row_cnt INTO _row_cnt FROM dblink( _iconnname, _sql ) AS ( row_cnt integer );
      -- Datensatz gibt es noch nicht.
      IF _row_cnt = 0 THEN
          --> Insert
          _sql := format( 'INSERT INTO egw_categories( cat_access, cat_appname, cat_name          , cat_description   , cat_data                       , last_mod ) ' ||
                          'VALUES                    ( %L        , %L         , %L                , %L                , %L                             , %L       ) ' ||
                          'RETURNING cat_id;',
                                                        'public'  , 'calendar' , 'Stundenausgleich', 'Stundenausgleich', '{"color":"#7f00ff","icon":""}', _now             -- INSERT
          );
          SELECT cat_id INTO _cat_id FROM dblink( _iconnname, _sql ) AS ( cat_id integer );
          PERFORM tsystem.settings__set( 'X2E_EGroupWare_Kategorie_Stundenausgleich', _cat_id::varchar(64) );
      -- Datensatz existiert schon.
      ELSE
          --> Set Setting
          _sql := format( 'SELECT min( cat_id ) FROM egw_categories WHERE cat_appname = %L AND cat_access = %L AND cat_name = %L', 'calendar', 'public', 'Stundenausgleich' );
          PERFORM tsystem.settings__set( 'X2E_EGroupWare_Kategorie_Stundenausgleich', cat_id )
          FROM dblink( _iconnname, _sql ) AS ( cat_id varchar(64) );
      END IF;

      -- Abwesend (dunkelgrau)
      _sql := format( 'SELECT count( * ) FROM egw_categories WHERE cat_appname = %L AND cat_access = %L AND cat_name = %L', 'calendar', 'public', 'Abwesend' );
      SELECT row_cnt INTO _row_cnt FROM dblink( _iconnname, _sql ) AS ( row_cnt integer );
      -- Datensatz gibt es noch nicht.
      IF _row_cnt = 0 THEN
          --> Insert
          _sql := format( 'INSERT INTO egw_categories( cat_access, cat_appname, cat_name          , cat_description   , cat_data                       , last_mod ) ' ||
                          'VALUES                    ( %L        , %L         , %L                , %L                , %L                             , %L       ) ' ||
                          'RETURNING cat_id;',
                                                        'public'  , 'calendar' , 'Abwesend'        , 'Abwesend'        , '{"color":"#666666","icon":""}', _now             -- INSERT
          );
          SELECT cat_id INTO _cat_id FROM dblink( _iconnname, _sql ) AS ( cat_id integer );
          PERFORM tsystem.settings__set( 'X2E_EGroupWare_Abwesend', _cat_id::varchar(64) );
      -- Datensatz existiert schon.
      ELSE
          --> Set Setting
          _sql := format( 'SELECT min( cat_id ) FROM egw_categories WHERE cat_appname = %L AND cat_access = %L AND cat_name = %L', 'calendar', 'public', 'Abwesend' );
          PERFORM tsystem.settings__set( 'X2E_EGroupWare_Abwesend', cat_id )
          FROM dblink( _iconnname, _sql ) AS ( cat_id varchar(64) );
      END IF;

      -- Schule ~ "außer Haus" (dunkelgrau)
      _sql := format( 'SELECT count( * ) FROM egw_categories WHERE cat_appname = %L AND cat_access = %L AND cat_name = %L', 'calendar', 'public', 'Schule' );
      SELECT row_cnt INTO _row_cnt FROM dblink( _iconnname, _sql ) AS ( row_cnt integer );
      -- Datensatz gibt es noch nicht.
      IF _row_cnt = 0 THEN
          --> Insert
          _sql := format( 'INSERT INTO egw_categories( cat_access, cat_appname, cat_name          , cat_description          , cat_data                       , last_mod ) ' ||
                          'VALUES                    ( %L        , %L         , %L                , %L                       , %L                             , %L       ) ' ||
                          'RETURNING cat_id;',
                                                        'public'  , 'calendar' , 'Schule'          , 'Schule ~ "außer Haus"'  , '{"color":"#666666","icon":""}', _now             -- INSERT
          );
          SELECT cat_id INTO _cat_id FROM dblink( _iconnname, _sql ) AS ( cat_id integer );
          PERFORM tsystem.settings__set( 'X2E_EGroupWare_Kategorie_Schule', _cat_id::varchar(64) );
      -- Datensatz existiert schon.
      ELSE
          --> Set Setting
          _sql := format( 'SELECT min( cat_id ) FROM egw_categories WHERE cat_appname = %L AND cat_access = %L AND cat_name = %L', 'calendar', 'public', 'Schule' );
          PERFORM tsystem.settings__set( 'X2E_EGroupWare_Kategorie_Schule', cat_id )
          FROM dblink( _iconnname, _sql ) AS ( cat_id varchar(64) );
      END IF;

      -- Schulungsmaßnahmen ~ "außer Haus" (dunkelgrau)
      _sql := format( 'SELECT count( * ) FROM egw_categories WHERE cat_appname = %L AND cat_access = %L AND cat_name = %L', 'calendar', 'public', 'Schulung' );
      SELECT row_cnt INTO _row_cnt FROM dblink( _iconnname, _sql ) AS ( row_cnt integer );
      -- Datensatz gibt es noch nicht.
      IF _row_cnt = 0 THEN
          --> Insert
          _sql := format( 'INSERT INTO egw_categories( cat_access, cat_appname, cat_name          , cat_description          , cat_data                       , last_mod ) ' ||
                          'VALUES                    ( %L        , %L         , %L                , %L                       , %L                             , %L       ) ' ||
                          'RETURNING cat_id;',
                                                        'public'  , 'calendar' , 'Schulung'        , 'Schulung ~ "außer Haus"', '{"color":"#666666","icon":""}', _now             -- INSERT
          );
          SELECT cat_id INTO _cat_id FROM dblink( _iconnname, _sql ) AS ( cat_id integer );
          PERFORM tsystem.settings__set( 'X2E_EGroupWare_Kategorie_Schulung', _cat_id::varchar(64) );
      -- Datensatz existiert schon.
      ELSE
          --> Set Setting
          _sql := format( 'SELECT min( cat_id ) FROM egw_categories WHERE cat_appname = %L AND cat_access = %L AND cat_name = %L', 'calendar', 'public', 'Schulung' );
          PERFORM tsystem.settings__set( 'X2E_EGroupWare_Kategorie_Schulung', cat_id )
          FROM dblink( _iconnname, _sql ) AS ( cat_id varchar(64) );
      END IF;

      -- Verbindung zu Egroupware-DB wieder schließen.
      PERFORM dblink_disconnect( _iconnname );
  END IF;
END; $$
/*
  SELECT TSystem.Settings__GetText( 'X2E_EGroupWare_Kategorie_Urlaub' ) AS Urlaub
      , TSystem.Settings__GetText( 'X2E_EGroupWare_Kategorie_Urlaub_einhalb' ) AS Urlaub2
      , TSystem.Settings__GetText( 'X2E_EGroupWare_Kategorie_Stundenausgleich' ) AS Stundenausgleich
      , TSystem.Settings__GetText( 'X2E_EGroupWare_Abwesend' ) AS Abwesend
      , TSystem.Settings__GetText( 'X2E_EGroupWare_Kategorie_Schule' ) AS Schule
      , TSystem.Settings__GetText( 'X2E_EGroupWare_Kategorie_Schulung' ) AS Schulung
  ;
*/
--

-- SCHEMA z_50_customer anlegen.
DROP SCHEMA IF EXISTS egroupware CASCADE;
CREATE SCHEMA IF NOT EXISTS z_50_customer;
--

-- TRIGGER bdea__a_iud__egroupware anlegen.
-- Aktualisiert die Abwesenheit im Kalender von eGroupware.
DROP TRIGGER IF EXISTS bdepab__a_iud__egroupware ON bdepab;
SELECT tsystem.function__drop_by_regex( _functionname_regex => 'bdepab__a_iud__egroupware', _commit => true );
CREATE OR REPLACE FUNCTION z_50_customer.bdepab__a_iud__egroupware() RETURNS TRIGGER AS $$
  DECLARE

    _now        integer;  -- Jetziger Zeitpunkt.
    _rndstr     text;     -- Zufällige und somit einzigartige Zeichenkette.
    _iconnname  text;     -- Verbindungsname für Verbindung zur Egroupware-DB, sollte einzigartig sein.
    _uidnumber  integer;  -- Hinterlegte eGroupware-UserID des von der Abwesenheit betroffenen Nutzers.
    -- Debug
    -- _prefix     varchar = format( '-> bdepab__a_iud__egroupware bdab_id:%L -', coalesce( new.bdab_id, old.bdab_id) );

  BEGIN
    -- Debug
    -- RAISE NOTICE 'call: bdepab__a_iud__egroupware(): tg_op:%, old:%, new:%;',
    --                                                  tg_op  , old  , new    ;
    -- Mindestpause, Pause, Raucherpause bei der Synchronisation von Abwesenheiten nach eGroupware ausschließen - Zuerst in new schauen für insert und update und dann erst in old für delete.
    IF ( coalesce( new.bdab_aus_id, old.bdab_aus_id, 0 ) NOT IN ( 101, 110, 103 ) ) THEN
        -- Hinterlegte eGroupware-UserID des von der Abwesenheit betroffenen Nutzers ermitteln.
        _uidnumber := TRecnoParam.GetInteger( 'uidnumber', llv.dbrid )
                       FROM llv
                      WHERE ll_minr = coalesce( new.bdab_minr, old.bdab_minr, 0 );
        -- Abwesenheiten nur noch von Nutzern mit hinterlegter eGroupware-UserID nach eGroupware exportieren.
        IF ( _uidnumber IS NOT null ) THEN
            -- Jetzigen Zeitpunkt ermitteln.
            _now := EXTRACT( EPOCH FROM ( currenttime()::timestamp at time zone 'Europe/Berlin' ) )::bigint;
            -- Zufällige Zeichenkette erstellen.
            _rndstr := array_to_string( array_cat_agg( array[ chr( 97 + round( random() * 25 )::int ) ] ), '' ) FROM generate_series( 1, 5 );
            -- Einzigartigen Verbindungsnamen für Verbindung zur Egroupware-DB kreieren.
            _iconnname := 'egconn_' || coalesce( new.dbrid, old.dbrid, 'null' ) || '_' || current_user || '_' || _rndstr;
            -- Debug
            -- RAISE NOTICE '% _iconnname:%;', _prefix, _iconnname;
            -- Verbindung zu Egroupware-DB aufbauen.
            PERFORM dblink_connect_u( _iconnname, tsystem.settings__gettext( 'X2E_EGroupWare_DBLink' ) );
            IF TG_OP='DELETE' THEN
                -- Gelöschte Abwesenheit in Egroupware als "gelöscht" markieren.
                PERFORM z_50_customer.egw_cal__delete( _cal_uid => ( 'prodat-' || old.dbrid )::varchar(128), _timestamp => _now, _connname => _iconnname );
            ELSE  -- TG_OP='INSERT' OR TG_OP='UPDATE'
                -- Geänderte Abwesehenheit in Egroupware anpassen.
                PERFORM z_50_customer.egw_cal__export( _cal_uid => ( 'prodat-' || new.dbrid )::varchar(128), _connname => _iconnname );
            END IF;
            -- Verbindung zu Egroupware-DB wieder schließen.
            PERFORM dblink_disconnect( _iconnname );
        -- Debug
        -- ELSE
        --     RAISE NOTICE '% skipping: uidnumber is null.', _prefix;
        END IF;
    -- Debug
    -- ELSE
    --     RAISE NOTICE '% skipping: pause (bdab_aus_id:%).', _prefix, coalesce( new.bdab_aus_id, old.bdab_aus_id, 0 );
    END IF;
    --
    IF ( TG_OP = 'DELETE' ) THEN
        RETURN old;
    ELSIF (TG_OP IN ( 'INSERT', 'UPDATE' ) ) THEN
        RETURN new;
    END IF;
  END $$ LANGUAGE plpgsql;
  --
  CREATE TRIGGER bdepab__a_iud__egroupware
    AFTER INSERT OR DELETE OR UPDATE OF bdab_minr, bdab_anf, bdab_anft, bdab_end, bdab_endt, bdab_aus_id, bdab_bdabb_id, bdab_bem
    ON bdepab
    FOR EACH ROW
    EXECUTE PROCEDURE z_50_customer.bdepab__a_iud__egroupware();
--

-- FUNCTION z_50_customer.egw_cal__delete anlegen.
-- In der zwischwenzeit in Prodat gelöschte Abwesenheiten auch in Egroupware auch löschen.
SELECT tsystem.function__drop_by_regex( _functionname_regex => 'egw_cal__delete', _commit => true );
CREATE OR REPLACE FUNCTION z_50_customer.egw_cal__delete(
    _cal_uid          varchar(128),
    _timestamp        integer,
    _connname         text DEFAULT null
) RETURNS integer AS $$
  DECLARE

      _rndstr    text;
      _iconnname text := _connname; -- eigener (interner) Verbindungsname zur Egroupware-DB.
      _row_cnt  integer;
      _cal_id   integer;
      _sql      text;
      -- Debug
      -- _prefix   varchar = format( '-> egw_cal__delete cal_uid:%L -', _cal_uid );

  BEGIN
      -- Debug
      -- RAISE notice 'call: egw_cal__delete( _cal_uid => %, _timestamp => %, _connname => % );',
      --                                      _cal_uid     , _timestamp     , _connname         ;
      -- Falls kein Verbindungsname übergeben wurde, ...
      IF _connname IS NULL THEN
          -- Zufällige und somit einzigartige Zeichenkette erstellen und ...
          _rndstr := array_to_string(
                          array_cat_agg(
                              array[ chr( 97 + round( random() * 25 )::int ) ]
                          ),''
                      )
                      FROM generate_series( 1, 5 );
          -- ... damit ein Verbindungsnamen erzeugen. ...
          _iconnname := 'egconn_' || _cal_uid::text || '_' || current_user || '_' || _rndstr;
          -- Debug
          -- RAISE NOTICE '% _iconnname:%;', _prefix, _iconnname;
          -- ... Dann eine eigene Verbindung zu Egroupware-DB aufbauen.
          PERFORM dblink_connect_u( _iconnname, tsystem.settings__gettext( 'X2E_EGroupWare_DBLink' ) );
      END IF;

      -- Eintrag in Tabelle egw_cal löschen.
      -- Prüfen, ob Datensatz mit der gegebenen cal_uid bereits existiert.
      _sql := format( 'SELECT count(*) FROM egw_cal WHERE cal_uid = %L;', _cal_uid );
      SELECT row_cnt INTO _row_cnt FROM dblink( _iconnname, _sql ) AS ( row_cnt integer );
      -- Datensatz existiert.
      IF _row_cnt > 0 THEN
          --> Delete
          _sql := format( 'UPDATE egw_cal SET cal_deleted = %L ' ||
                          'WHERE cal_uid = %L ' ||
                          'RETURNING cal_id;',
                                              _timestamp, -- Update
                                  _cal_uid                 -- WHERE
          );
          -- Debug
          -- RAISE NOTICE '% sql:%', _prefix, _sql;
          SELECT cal_id INTO _cal_id FROM dblink( _iconnname, _sql ) AS ( cal_id integer );
      END IF;

      -- Falls kein Verbindungsname übergeben wurde, ...
      IF _connname IS NULL THEN
          -- ... dann eine eigene Verbindung zu Egroupware-DB wieder schließen.
          PERFORM dblink_disconnect( _iconnname );
      END IF;

      RETURN _cal_id;

  END $$ LANGUAGE plpgsql VOLATILE;
--

-- FUNCTION z_50_customer.egw_cal__upsert anlegen.
-- Neue Abwesenheiten in Egroupware anlegen und bereits existierende anpassen.
SELECT tsystem.function__drop_by_regex( _functionname_regex => 'egw_cal__upsert', _commit => true );
CREATE OR REPLACE FUNCTION z_50_customer.egw_cal__upsert(
    _cal_uid          varchar(128),
    _cal_owner        integer,
    _cal_category     varchar(64),
    _range_start      bigint,
    _range_end        bigint,
    _cal_title        varchar(255),
    _cal_description  text,
    _cal_modifier     integer,
    _cal_modified     bigint,
    _cal_non_blocking smallint,
    _cal_etag         integer,
    _cal_creator      integer,
    _cal_created      bigint,
    _tz_id            integer,
    _cal_recur_date   integer,
    _cal_user_type    varchar(1),
    _cal_status       varchar(1),
    _cal_quantity     integer,
    _cal_role         varchar(64),
    _recur_exception  boolean,
    _connname         text DEFAULT null
) RETURNS integer AS $$
  DECLARE

      _rndstr    text;
      _iconnname text := _connname; -- eigener (interner) Verbindungsname zur Egroupware-DB.
      _row_cnt  integer;
      _cal_id   integer;
      _sql      text;
      -- Debug
      -- _prefix   varchar = format( '-> egw_cal__upsert cal_uid:%L -', _cal_uid );

  BEGIN
      -- Debug
      -- RAISE notice 'call: egw_cal__upsert( _cal_uid => %, _cal_owner => %, _cal_category => %, _range_start => %, _range_end => %, _cal_title => %, _cal_description => %, _cal_modifier => %, _cal_modified => %, _cal_non_blocking => %, _cal_etag => %, _cal_creator => %, _cal_created => %, _tz_id => %, _cal_recur_date => %, _cal_user_type => %, _cal_status => %, _cal_quantity => %, _cal_role => %, _recur_exception => %, _connname => % );',
      --                                      _cal_uid     , _cal_owner     , _cal_category     , _range_start     , _range_end     , _cal_title     , _cal_description     , _cal_modifier     , _cal_modified     , _cal_non_blocking     , _cal_etag     , _cal_creator     , _cal_created     , _tz_id     , _cal_recur_date     , _cal_user_type     , _cal_status     , _cal_quantity     , _cal_role     , _recur_exception     , _connname         ;
      -- Falls kein Verbindungsname übergeben wurde, ...
      IF _connname IS NULL THEN
          -- Zufällige und somit einzigartige Zeichenkette erstellen.
          _rndstr := array_to_string(
                          array_cat_agg(
                              array[ chr( 97 + round( random() * 25 )::int ) ]
                          ),''
                      )
                      FROM generate_series( 1, 5 );
          -- ... damit ein Verbindungsnamen erzeugen. ...
          _iconnname := 'egconn_' || _cal_uid::text || '_' || current_user || '_' || _rndstr;
          -- Debug
          -- RAISE NOTICE '% _iconnname:%;', _prefix, _iconnname;
          -- ... Dann eine eigene Verbindung zu Egroupware-DB aufbauen.
          PERFORM dblink_connect_u( _iconnname, tsystem.settings__gettext( 'X2E_EGroupWare_DBLink' ) );
      END IF;

      -- Eintrag in Tabelle egw_cal schreiben bzw. anpassen.
      -- Prüfen, ob Datensatz mit der gegebenen cal_uid bereits existiert.
      _sql := format( 'SELECT count(*) FROM egw_cal WHERE cal_uid = %L;', _cal_uid );
      SELECT row_cnt INTO _row_cnt FROM dblink( _iconnname, _sql ) AS ( row_cnt integer );
      -- Datensatz gibt es noch nicht.
      IF _row_cnt = 0 THEN
          --> Insert
          _sql := format( 'INSERT INTO egw_cal( cal_uid , cal_owner     , cal_category     , range_start     , range_end     , cal_title     , cal_description     , cal_modifier     , cal_modified     , cal_non_blocking     , cal_etag     , cal_creator     , cal_created     , tz_id     ) ' ||
                          'VALUES             ( %L      , %L            , %L               , %L              , %L            , %L            , %L                  , %L               , %L               , %L                   , %L           , %L              , %L              , %L        ) ' ||
                          'RETURNING cal_id;',
                                                _cal_uid, _cal_owner    , _cal_category    , _range_start    , _range_end    , _cal_title    , _cal_description    , _cal_modifier    , _cal_modified    , _cal_non_blocking    , _cal_etag    , _cal_creator    , _cal_created    , _tz_id       -- INSERT
          );
      -- Datensatz existiert schon.
      ELSE
          --> Update
          _sql := format( 'UPDATE egw_cal SET             cal_owner = %L, cal_category = %L, range_start = %L, range_end = %L, cal_title = %L, cal_description = %L, cal_modifier = %L, cal_modified = %L, cal_non_blocking = %L, cal_etag = %L,                                     tz_id = %L, cal_deleted = NULL ' ||
                          'WHERE cal_uid = %L '                                                                                                                                                                                                                                                                       ||
                          'RETURNING cal_id;',
                                                          _cal_owner    , _cal_category    , _range_start    , _range_end    , _cal_title    , _cal_description    , _cal_modifier    , _cal_modified    , _cal_non_blocking    , _cal_etag    ,                                     _tz_id     , -- UPDATE
                                    _cal_uid                                                                                                                                                                                                                                                      -- WHERE
          );
      END IF;
      -- Debug
      -- RAISE NOTICE '% sql:%', _prefix, _sql;
      SELECT cal_id INTO _cal_id FROM dblink( _iconnname, _sql ) AS ( cal_id integer );

      -- Eintrag in Tabelle egw_cal_user schreiben bzw. anpassen.
      -- Prüfen, ob Datensatz mit der Kombination aus vorher zurückgegebenen cal_id und Benutzer (cal_owner/ cal_user_id) bereits existiert.
      _sql := format( 'SELECT count(*) FROM egw_cal_user WHERE cal_id = %L AND cal_user_id = %L;', _cal_id, _cal_owner );
      SELECT row_cnt INTO _row_cnt FROM dblink( _iconnname, _sql ) AS ( row_cnt integer );
      -- Datensatz gibt es noch nicht.
      IF _row_cnt = 0 THEN
          --> Insert
          _sql := format( 'INSERT INTO egw_cal_user( cal_id            , cal_recur_date     , cal_user_type     , cal_user_id     , cal_status     , cal_quantity     , cal_role      ) ' ||
                          'VALUES                  ( %L                , %L                 , %L                , %L              , %L             , %L               , %L            );',
                                                    _cal_id            , _cal_recur_date    , _cal_user_type    , _cal_owner      , _cal_status    , _cal_quantity    , _cal_role       -- INSERT
          );
      -- Datensatz existiert schon.
      ELSE
          --> Update
          _sql := format( 'UPDATE egw_cal_user SET                       cal_recur_date = %L, cal_user_type = %L,                   cal_status = %L, cal_quantity = %L, cal_role = %L '   ||
                          'WHERE cal_id = %L AND cal_user_id = %L;',
                                                                          _cal_recur_date   , _cal_user_type   ,                   _cal_status    , _cal_quantity    , _cal_role     ,   -- UPDATE
                                  _cal_id  ,     _cal_owner                                                                                                                                     -- WHERE
          );
      END IF;
      -- Debug
      -- RAISE NOTICE '% sql:%', _prefix, _sql;
      PERFORM * FROM dblink_exec( _iconnname, _sql );

      -- Eintrag in Tabelle egw_cal_dates schreiben bzw. anpassen.
      -- Prüfen, ob Datensatz mit der vorher zurückgegebenen cal_id bereits existiert.
      _sql := format( 'SELECT count(*) FROM egw_cal_dates WHERE cal_id = %L;', _cal_id );
      SELECT row_cnt INTO _row_cnt FROM dblink( _iconnname, _sql ) AS ( row_cnt integer );
      -- Datensatz gibt es noch nicht.
      IF _row_cnt = 0 THEN
          --> Insert
          _sql := format( 'INSERT INTO egw_cal_dates( cal_id     , cal_start     , cal_end     , recur_exception      ) ' ||
                          'VALUES                   ( %L         , %L            , %L          , %L                   );' ,
                                                      _cal_id    , _range_start  , _range_end  , _recur_exception       -- INSERT
          );
      -- Datensatz existiert schon.
      ELSE
          --> Update
          _sql := format( 'UPDATE egw_cal_dates SET                cal_start = %L, cal_end = %L, recur_exception = %L '   ||
                          'WHERE cal_id = %L;',
                                                                    _range_start  , _range_end  , _recur_exception     , -- UPDATE
                                  _cal_id                                                                                -- WHERE
          );
      END IF;
      -- Debug
      -- RAISE NOTICE '% sql:%', _prefix, _sql;
      PERFORM * FROM dblink_exec( _iconnname, _sql );

      -- Falls kein Verbindungsname übergeben wurde, ...
      IF _connname IS NULL THEN
          -- ... dann eine eigene Verbindung zu Egroupware-DB wieder schließen.
          PERFORM dblink_disconnect( _iconnname );
      END IF;

      RETURN _cal_id;

  END $$ LANGUAGE plpgsql VOLATILE;
--

-- FUNCTION z_50_customer.egw_cal__export anlegen.
-- Abwesenheiten von Prodat nach EGroupware synchronisieren.
SELECT tsystem.function__drop_by_regex( _functionname_regex => 'egw_cal__export', _commit => true );
CREATE OR REPLACE FUNCTION z_50_customer.egw_cal__export(
    _time_range_start   date DEFAULT ( CURRENT_DATE - interval '1 month' )::date,
    _time_range_end     date DEFAULT ( CURRENT_DATE                      )::date,
    _cal_uid            varchar(128) DEFAULT null,
    _connname           text DEFAULT null
) RETURNS VOID AS $$
  DECLARE

    _rndstr    text;
    _iconnname text := _connname; -- eigener (interner) Verbindungsname zur Egroupware-DB.
    _now integer := EXTRACT( EPOCH FROM ( currenttime()::timestamp at time zone 'Europe/Berlin' ) )::bigint;
    _sql text;
    -- Debug
    -- _prefix   varchar = format( '-> egw_cal__export -' );

  BEGIN
      -- Debug
      -- RAISE notice 'call: egw_cal__export( _time_range_start => %, _time_range_end => %, _cal_uid => %, _connname => % );',
      --                                      _time_range_start     , _time_range_end     , _cal_uid     , _connname         ;
      -- Falls kein Verbindungsname übergeben wurde, ...
      IF _connname IS NULL THEN
          _rndstr := array_to_string(
                    array_cat_agg(
                        array[ chr( 97 + round( random() * 25 )::int ) ]
                    ),''
                )
                FROM generate_series( 1, 5 );
          -- ... damit ein Verbindungsnamen erzeugen. ...
          _iconnname := 'egconn_' || coalesce( _cal_uid::text, to_char( _time_range_start, 'YYYY-MM-DD' ) ) || '_' || current_user || '_' || _rndstr;
          -- Debug
          -- RAISE NOTICE '% _iconnname:%;', _prefix, _iconnname;
          -- ... Dann eine eigene Verbindung zu Egroupware-DB aufbauen.
          PERFORM dblink_connect_u( _iconnname, tsystem.settings__gettext( 'X2E_EGroupWare_DBLink' ) );
      END IF;

      -- Keine konkrete cal_uid als Parameter übergeben, Zeitbereich wird verwendet.
      IF _cal_uid IS null THEN
          -- Debug
          -- RAISE NOTICE '% Weggefallene Abwesenheiten im Zeitfenster % ~> % löschen;', _prefix, _time_range_start, _time_range_end;
          -- SQL-Abfrage zusammenbauen, um alle im Synchronirungszeitraum in eGroupware existierenden cal_uids zu ermittlen.
          _sql := format( 'SELECT split_part( cal_uid, %L, 2 ) AS dbrid '                                                         || -- nur Teil nach Prefix, entspricht bdepab.dbrid
                          '  FROM egw_cal '                                                                                       ||
                          ' WHERE cal_uid ~ %L '                                                                                  || -- Filter auf Prefix gefolgt von ganzen Zahlen
                          '   AND cal_deleted IS null '                                                                           ||
                          '   AND range_end   > EXTRACT( EPOCH FROM ( %L )::timestamp AT time zone ''Europe/Berlin'' )::bigint '  || -- Parameter _time_range_start -- der Zeitraum der Abwesenheit überlappt mit dem Synchronirungszeitraum
                          '   AND range_start < EXTRACT( EPOCH FROM ( %L )::timestamp AT time zone ''Europe/Berlin'' )::bigint;' ,   -- Paremeter _time_range_end
                          'prodat-',           -- Prefix
                          '^prodat-[0-9]+$',   -- Filter auf Prefix gefolgt von ganzen Zahlen
                          _time_range_start,  -- Parameter _time_range_start
                          _time_range_end     -- Paremeter _time_range_end
          );
          -- Die im angegebenen Synchronirungszeitraum in Prodat gelöschten Abwesenheiten jetzt auch in EGroupware löschen.
          PERFORM z_50_customer.egw_cal__delete( _cal_uid => ( 'prodat-' || sub.dbrid )::varchar(128), _timestamp => _now, _connname => _iconnname )
              FROM (
                      -- Die im angegebenen Synchronisierungszeitraum in eGroupware, aber nicht mehr in Prodat existierenden cal_uids bzw. dbrids ermittlen.
                      SELECT dbrid FROM dblink( tsystem.settings__gettext('X2E_EGroupWare_DBLink'), _sql ) AS ( dbrid varchar(128) )

                      EXCEPT

                      SELECT bdepab.dbrid::varchar(128)
                        FROM bdepab
                        JOIN llv ON bdab_minr = ll_minr
                      WHERE bdab_aus_id NOT IN (101, 110, 103)
                        AND TRecnoParam.GetInteger( 'uidnumber', llv.dbrid ) IS NOT null
                        AND bdab_end > _time_range_start -- der Zeitraum der Abwesenheit überlappt mit dem Synchronirungszeitraum
                        AND bdab_anf < _time_range_end
                  ) AS sub;
      END IF;

      -- Debug
      -- IF ( _cal_uid IS NOT null ) THEN
      --     RAISE NOTICE '% Abwesenheit (cal_uid:%) synchronisieren;', _prefix, _cal_uid;
      -- ELSE
      --     RAISE NOTICE '% Abwesenheiten im Zeitfenster % ~> % synchronisieren;', _prefix, _time_range_start, _time_range_end;
      -- END IF;
      -- Neue Abwesenheiten erstellen oder bereits bestehenende anpassen.
      PERFORM
        z_50_customer.egw_cal__upsert
        (
            _cal_uid            => src.cal_uid,
            _cal_owner          => src.cal_owner,
            _cal_category       => src.cal_category,
            _range_start        => src.range_start,
            _range_end          => src.range_end,
            _cal_title          => src.cal_title,
            _cal_description    => src.cal_description,
            _cal_modifier       => src.cal_modifier,
            _cal_modified       => src.cal_modified,
            _cal_non_blocking   => src.cal_non_blocking,
            _cal_etag           => src.cal_etag,
            _cal_creator        => src.cal_creator,
            _cal_created        => src.cal_created,
            _tz_id              => src.tz_id,
            _cal_recur_date     => src.cal_recur_date,
            _cal_user_type      => src.cal_user_type,
            _cal_status         => src.cal_status,
            _cal_quantity       => src.cal_quantity,
            _cal_role           => src.cal_role,
            _recur_exception    => src.recur_exception,
            _connname           => _iconnname
        )
      FROM
        (
            SELECT
              ( 'prodat-' || bdepab.dbrid )::varchar(128)                                                                         AS cal_uid,           -- eindeutiger Ident der Abwesenheit PRODAT
              TRecnoParam.GetInteger( 'uidnumber', llv.dbrid )::integer                                                           AS cal_owner,         -- Inhalt uidNumber über Erweiterungsfeld PRODAT
                                                                                                                                                        -- Es kann perspektivisch auch die uidNumber
                                                                                                                                                        -- als PRODAT Mitarbeiter-Nr verwendet werden
                                                                                                                                                        -- uidNumber aus Active-Directory des Bearbeiters
              CASE
                WHEN
                  bdab_aus_id = 1   THEN COALESCE( NULLIF( TSystem.Settings__GetText( 'X2E_EGroupWare_Kategorie_Urlaub' ), '' ), 'urlaub' )                     -- Urlaub
                WHEN
                  bdab_aus_id = 100 THEN COALESCE( NULLIF( TSystem.Settings__GetText( 'X2E_EGroupWare_Kategorie_Urlaub_einhalb' ), '' ), 'urlaub05' )           -- Urlaub 1/2
                WHEN
                  bdab_aus_id = 104 THEN COALESCE( NULLIF( TSystem.Settings__GetText( 'X2E_EGroupWare_Kategorie_Schule' ), '' ), 'schule' )                     -- Schule ~ "außer Haus"
                WHEN
                  bdab_aus_id = 108 THEN COALESCE( NULLIF( TSystem.Settings__GetText( 'X2E_EGroupWare_Kategorie_Schulung' ), '' ), 'schulung' )                 -- Schulungsmaßnahmen ~ "außer Haus"
                WHEN
                  bdab_aus_id = 150 THEN COALESCE( NULLIF( TSystem.Settings__GetText( 'X2E_EGroupWare_Kategorie_Stundenausgleich' ), '' ), 'stundenausgleich' ) -- Stundenausgleich
                ELSE
                  COALESCE( NULLIF( TSystem.Settings__GetText( 'X2E_EGroupWare_Abwesend' ), '' ), 'abwesend' )                                                  -- 'Abwesend'
              END::varchar(64)                                                                                                    AS cal_category,      -- Abwesenheitsgrund, Spalte "cat_id" der Tablle "egw_categories".

              EXTRACT(
                EPOCH FROM (
                  bdab_anf + coalesce( bdab_anft,  '0:00:00'::time )
                )::timestamp at time zone 'Europe/Berlin'
              )::bigint                                                                                                           AS range_start,       -- Anfang der Abwesenheit, Zeitzone beachten siehe https://web.archive.org/web/20150308140507/http://icu.iorahealth.com/blog/2012/05/07/expressing-postgresql-timestamps-without-zones-in-local-timee/
              EXTRACT(
                EPOCH FROM (
                  bdab_end + coalesce( bdab_endt, '23:59:59'::time )
                )::timestamp at time zone 'Europe/Berlin'
              )::bigint                                                                                                           AS range_end,         -- Ende der Abwesenheit, Zeitzone beachten siehe https://web.archive.org/web/20150308140507/http://icu.iorahealth.com/blog/2012/05/07/expressing-postgresql-timestamps-without-zones-in-local-timee/

              COALESCE(
                (
                    CASE WHEN bdab_aus_id NOT IN ( 1, 100, 104, 108, 150 ) THEN 'Abwesenheit' ELSE ab_txt END || ' ' ||                                 -- Abwesenheitsgrund +
                    ad_vorn || ' ' || ad_name                                                                                                           -- voller Name
                ),
                '1'
              )::varchar(255)                                                                                                     AS cal_title,         -- Titel des Kalendereintrags (Abweseheitsgrund, Vor- und Zuname des Mitarbeiters)
              COALESCE(
                bdabb_bem,                                                                                                                              -- aus Beantragung ...
                bdab_bem                                                                                                                                -- ansosnsten aus Bestädigung ...
              )::text                                                                                                             AS cal_description,   -- Bemerkungen zur Abwesenheit

              COALESCE(
                TRecnoParam.GetInteger( 'uidnumber', llmodifiedby.dbrid ),
                llmodifiedby.minr,
                TRecnoParam.GetInteger( 'uidnumber', llv.dbrid )
              )::integer                                                                                                          AS cal_modifier,      -- analog cal_owner mit Fallback auf Mitarbeiter-Nr
              _now                                                                                                                AS cal_modified,      -- Änderungszeitpunkt

              0::smallint                                                                                                         AS cal_non_blocking,  -- fix, alle Termine sind immer blockierend
              0::integer                                                                                                          AS cal_etag,          -- fix, Anwendungsfall nicht klar, vorerst 0
              COALESCE(
                TRecnoParam.GetInteger( 'uidnumber', llinsertby.dbrid ),
                llinsertby.minr,
                TRecnoParam.GetInteger( 'uidnumber', llv.dbrid )
              )::integer                                                                                                          AS cal_creator,       -- analog cal_owner Ersteller der Abwesenheit
              _now                                                                                                                AS cal_created,       -- Erstellungszeitpunkt

              322::integer                                                                                                        AS tz_id,             -- Zeitzone, Spalte "tz_id" aus der Tabelle "egw_cal_timezones", wo tz_tzid = 'Europe/Berlin'.
              0::integer                                                                                                          AS cal_recur_date,    -- Tabelle egw_cal_user
              'u'::varchar(1)                                                                                                     AS cal_user_type,     -- Tabelle egw_cal_user; Default u; u=account
              'A'::varchar(1)                                                                                                     AS cal_status,        -- Tabelle egw_cal_user; Default A; U=unknown, A=accepted, R=rejected, T=tentative
              1::INTEGER                                                                                                          AS cal_quantity,      -- Tabelle egw_cal_user; Default 1; only for certain types (eg. resources)
              'REQ-PARTICIPANT'::varchar(64)                                                                                      AS cal_role,          -- Tabelle egw_cal_user; Default REQ-PARTICIPANT; CHAIR, REQ-PARTICIPANT, OPT-PARTICIPANT, NON-PARTICIPANT, X-CAT-$cat_id
              false::boolean                                                                                                      AS recur_exception    -- Tabelle egw_cal_dates; default ''; date is an exception

              -- Übergabe in custom Feld egroupware
              -- bdepab.dbrid AS cal_prodat_uid                                                         -- eindeutiger Ident der Abwesenheit
                                                                                                        -- anderen namen vergeben, wenn in egroupware angelegt

              --cal_priority smallint NOT NULL DEFAULT 2,                                                 -- wenn abweichend vom default, dann muss x2e Wunsch angeben
              --cal_public smallint NOT NULL DEFAULT 1,                                                   -- wenn abweichend vom default, dann muss x2e Wunsch angeben
              --cal_location character varying(255)                                                       -- kein Feld in PRODAT, ausser für Schulungen
              --cal_reference INTEGER NOT NULL DEFAULT 0,                                                 -- wenn Inhalt gewünscht, dann muss x2e Wunsch angeben
              --cal_special smallint DEFAULT 0,                                                           -- wenn Inhalt gewünscht, dann muss x2e Wunsch angeben werden
            FROM
              bdepab
              JOIN bdeabgruende ON ab_id = bdab_aus_id
              JOIN llv ON bdab_minr = ll_minr
              JOIN adk ON ll_ad_krz = ad_krz
              -- Letzter Bearbeiter der Abwesenheit
              LEFT JOIN LATERAL( SELECT dbrid, ll_minr AS minr FROM llv WHERE bdepab.modified_by = ll_db_usename ) AS llmodifiedby ON true
              -- Ersteller der Abwesenheit
              LEFT JOIN LATERAL( SELECT dbrid, ll_minr AS minr FROM llv WHERE bdepab.insert_by = ll_db_usename ) AS llinsertby ON true
              LEFT JOIN bdepabbe ON bdabb_id = bdab_bdabb_id
            WHERE bdab_aus_id NOT IN (101, 110, 103)                                                              -- Mindestpause, Pause, Raucherpause
              AND TRecnoParam.GetInteger( 'uidnumber', llv.dbrid ) IS NOT null                                    -- Abwesenheiten nur noch von Nutzern mit hinterlegter eGroupware-UserID nach eGroupware exportieren.
              AND (
                        ( _cal_uid IS NOT null AND ( 'prodat-' || bdepab.dbrid )::varchar(128) = _cal_uid )       -- eine kongrete Abwesenheit (dbrid)
                      OR
                        ( _cal_uid IS null AND bdab_end >= _time_range_start AND bdab_anf <= _time_range_end )    -- der Zeitraum der Abwesenheit überlappt mit dem Synchronirungszeitraum
                  )
        ) AS src;

      -- Falls kein Verbindungsname übergeben wurde, ...
      IF _connname IS NULL THEN
          -- ... dann eine eigene Verbindung zu Egroupware-DB wieder schließen.
          PERFORM dblink_disconnect( _iconnname );
      END IF;
  END $$ LANGUAGE plpgsql VOLATILE;
--

--- Ende eGroupware-Schnittstelle (#16795, #20362)