-- ###############################################################################
 -- Funktionen für den LOLL-EDI für die Bestellbestätigungen von Ceratizit

-- Generische Tabelle zum Einlesen von komplexen XML-EDIs
---- nur echt mit Rechtschreibefehler
CREATE TABLE IF NOT EXISTS edi_incomming_xml
(
    edi_id serial PRIMARY KEY,                         -- technische ID
    edi_type character varying(40) NOT NULL,           -- identifiziert den EDI-Typ
    edi_origin character varying(200) NOT NULL,        -- identifiziert die einzelne EDI-Message
    edi_content xml NOT NULL,                          -- der eigentliche EDI-Inhalt
    edi_insert timestamp without time zone NOT NULL,   -- Zeitpunkt des Eintreffens der EDI-Message
    edi_processing timestamp without time zone,        -- Zeitpunkt der letzten Verarbeitung der EDI-Message (erfolgreich oder nicht)
    edi_error_message text,                            -- Fehlermeldung der letzten Verarbeitung
    edi_success character varying(200)                 -- bei erfolgreicher Verarbeitung erzeugtes PRODAT-Artefakt
);

CREATE UNIQUE INDEX IF NOT EXISTS edi_incomming_xml__edi_type__edi_origin__uq
    ON edi_incomming_xml (edi_type, edi_origin);



 -- Typen zum Auslesen der LOLL-Bestellbestätigung

-- Ab- und Zuschläge einer Bestellposition
CREATE TYPE z_50_customer.loll__cera__ldsdok__pos__abzu AS (
  ldaz_txt varchar,                        -- Abzuschlagsbezeichnung
  ldaz_betrag numeric,                     -- Abzuschlagsbetrag
  ldaz_proz numeric                        -- Abzuschlag in Prozent
);

-- Bestellposition
CREATE TYPE z_50_customer.loll__cera__ldsdok__pos AS (
  ld_pos integer,                          -- Positionsnummer
  ld_aknr varchar,                         -- Artikelnummer
  ld_akbez varchar,                        -- Artikelbezeichnung
  ld_stk numeric,                          -- Menge
  ld_me varchar,                           -- Mengeneinheit
  ld_netto_grund numeric,                  -- Nettopreis vor Ab- und Zuschlägen
  ld_netto numeric,                        -- Nettopreis
  abzus z_50_customer.loll__cera__ldsdok__pos__abzu[],    -- Ab- und Zuschläge
  ld_ekref varchar                         -- Kundenreferenz
);

-- Rechnungs- und Lieferadresse
CREATE TYPE z_50_customer.loll__cera__adk AS (
  ad_fa1 varchar,                          -- Name
  ad_str varchar,                          -- Straße + Hausnummer
  ad_ort varchar,                          -- Ort
  ad_plz varchar,                          -- PLZ
  ad_landiso varchar                       -- Land
);

-- Bestellung
CREATE TYPE z_50_customer.loll__cera__ldsdok AS (
  ld_datum date,                           -- Bestelldatum
  ld_term date,                            -- Wunschtermin
  ld_post6 varchar,                        -- Belegnummer
  ld_post7 varchar,                        -- Message-ID
  ld_waer varchar,                         -- Währung
  ld_steuproz numeric,                     -- Mehrwertsteuersatz
  ld_steu numeric,                         -- Mehrwertsteuerbetrag
  ld_netto numeric,                        -- Nettobetrasg der Rechnung
  ld_kn varchar,                           -- Bestelladresse
  ld_krzl z_50_customer.loll__cera__adk,   -- Lieferadresse
  ld_krzf z_50_customer.loll__cera__adk,   -- Rechnungsadresse
  a2_sks numeric,                          -- Skontosatz
  a2_skv integer,                          -- Skontoverfall in Tagen
  a2_zak integer,                          -- Zahlungsziel in Tagen
  ld_poss z_50_customer.loll__cera__ldsdok__pos[],   -- Bestellpositionen
  ld_pos_anz integer                       -- Anzahl der bestellten Positionen
);

-- Funktionen zum Import von Bestellbestätigungen von Ceratizit

-- Code: Funktion zum XML-Parsen
CREATE OR REPLACE FUNCTION z_50_customer.xml__node__value__read( _xml_path varchar, _xml_content xml, _field_mandatory boolean = true ) RETURNS varchar AS $$
DECLARE
  _x xml[];
BEGIN

  -- liest aus einem XML den Wert eines Knotens aus
  -- Bei mehereren Knoten gleichen Namens wird nur der erste Wert zurückgegeben.
  -- Optional kann bei fehlendem oder leerem Knoten eine Exception geworfen werden, ansonsten käme null zurück.

  _x := xpath( _xml_path || '/text()', _xml_content );
  IF array_length( _x, 1 ) IS null AND _field_mandatory THEN
    RAISE EXCEPTION 'Knoten nicht gefunden oder leer: %', _xml_path;
  END IF;

  RETURN _x[1];

END $$ language plpgsql STRICT STABLE;
--

-- Code: Sucht eine LOLL-Adresse aus einem Ceratizit-Bestellbestätigungs-XML
-- Der Paramater _tag bezeichnet das XML-Attribut, welches die Adressarten unterscheidet.
CREATE OR REPLACE FUNCTION z_50_customer.loll__cera__adk__xml__read( _tag varchar, _xml xml ) RETURNS z_50_customer.loll__cera__adk AS $$
DECLARE
  _adk z_50_customer.loll__cera__adk;
  _xpath varchar;
  _ad_fa1_1 varchar;
  _ad_fa1_2 varchar;
BEGIN

  _xpath := '/IDOC/E1EDKA1[PARVW/text()=''' || _tag || ''']';
  _ad_fa1_1 := z_50_customer.xml__node__value__read( _xpath || '/NAME1', _xml );
  _ad_fa1_2 := z_50_customer.xml__node__value__read( _xpath || '/NAME2', _xml, false );
  _adk.ad_fa1     := _ad_fa1_1 || coalesce( _ad_fa1_2, '' );
  _adk.ad_str     := z_50_customer.xml__node__value__read( _xpath || '/STRAS', _xml );
  _adk.ad_plz     := z_50_customer.xml__node__value__read( _xpath || '/PSTLZ', _xml );
  _adk.ad_ort     := z_50_customer.xml__node__value__read( _xpath || '/ORT01', _xml );
  _adk.ad_landiso := z_50_customer.xml__node__value__read( _xpath || '/LAND1', _xml );

  RETURN _adk;

END $$ LANGUAGE plpgsql STRICT IMMUTABLE;
--

-- Hilfsfunktion zur Interpretation von "+" und "-"
CREATE OR REPLACE FUNCTION z_50_customer.loll__sign__get( _tag varchar, _xml xml ) RETURNS integer AS $$
DECLARE
  _sign varchar;
BEGIN

  -- liest aus einem Ceratizit-XML ein Vorzeichen aus
  -- Rückgabe: '+' => 1, '-' => -1, ansonsten Exception

  _sign := z_50_customer.xml__node__value__read( _tag, _xml  );
  CASE _sign
    WHEN '+' THEN RETURN 1;
    WHEN '-' THEN RETURN -1;
  ELSE
    RAISE EXCEPTION 'Unklares Vorzeichen: <%> Knoten %', _sign, _tag;
  END CASE;

END $$ LANGUAGE plpgsql STRICT IMMUTABLE;
--

-- Auslesen der Rohdaten der Bestellbestätigung aus einer XML, die von Ceratizit an LOLL übermittelt wurde
CREATE OR REPLACE FUNCTION z_50_customer.loll__cera__ldsdok__xml__read( _xml_content xml ) RETURNS z_50_customer.loll__cera__ldsdok AS $$
DECLARE
  _ldsdok z_50_customer.loll__cera__ldsdok;
  _xml xml;
  _positionen xml[];
  _position xml;
  _ldsdok_pos z_50_customer.loll__cera__ldsdok__pos;
  _abzus xml[];
  _abzu xml;
  _ldsdok_pos_abzu z_50_customer.loll__cera__ldsdok__pos__abzu;
  _sign integer;
BEGIN

  _xml := ( xpath( '/ORDERS05/IDOC', _xml_content ))[1];   -- Wrapper abschneiden
  _ldsdok.ld_post6    := z_50_customer.xml__node__value__read( '/IDOC/E1EDK02[QUALF/text()=''002'']/BELNR', _xml );  -- Belegnummer

  -- Markierung des zu importierenden Belegs zur beseeren Zuordnung von Fehlermeldungen
  RAISE NOTICE 'Ceratizit-Bestellbestätigung wird importiert - Beleg Nummer %', _ldsdok.ld_post6;

  _ldsdok.ld_kn       := 'WNT';                                                                                      -- Lieferantenadresse, Cerazitit DE Daimlerstr.
  _ldsdok.ld_krzl     := z_50_customer.loll__cera__adk__xml__read( 'WE', _xml );                                     -- Lieferadresse
  _ldsdok.ld_krzf     := z_50_customer.loll__cera__adk__xml__read( 'RE', _xml );                                     -- Rechnungsadresse
  _ldsdok.ld_datum    := z_50_customer.xml__node__value__read( '/IDOC/E1EDK03[IDDAT=''022'']/DATUM', _xml );         -- Bestelldatum
  _ldsdok.ld_term     := z_50_customer.xml__node__value__read( '/IDOC/E1EDK03[IDDAT=''002'']/DATUM', _xml );         -- Wunschtermin
  _ldsdok.ld_post7    := z_50_customer.xml__node__value__read( '/IDOC/EDI_DC40/SERIAL', _xml );                      -- Bestell-ID
  _ldsdok.ld_waer     := z_50_customer.xml__node__value__read( '/IDOC/E1EDK01/CURCY', _xml );                        -- Währung
  _ldsdok.ld_steuproz := z_50_customer.xml__node__value__read( '/IDOC/E1EDK04/MSATZ', _xml );                        -- Mehrwertsteuersatz
  _ldsdok.ld_steu     := z_50_customer.xml__node__value__read( '/IDOC/E1EDK04/MWSBT', _xml );                        -- Mehrwertsteuerbetrag
  _ldsdok.ld_netto    := z_50_customer.xml__node__value__read( '/IDOC/E1EDK05/BETRG', _xml );                        -- Rechnungsnettobetrag
  _ldsdok.a2_sks      := z_50_customer.xml__node__value__read( '/IDOC/E1EDK18[PRZNT > 0]/PRZNT', _xml, false );      -- Skonto in Prozent
  _ldsdok.a2_skv      := z_50_customer.xml__node__value__read( '/IDOC/E1EDK18[PRZNT > 0]/TAGE', _xml, false );       -- Skontoverfall in Tagen
  _ldsdok.a2_zak      := z_50_customer.xml__node__value__read( '/IDOC/E1EDK18[PRZNT = 0]/TAGE', _xml, false );       -- Zahlungsziel in Tagen
  _ldsdok.ld_pos_anz  := z_50_customer.xml__node__value__read( '/IDOC/E1EDS01[SUMID = ''001'']/SUMME', _xml );       -- Anzahl der Bestellpositionen

  -- Bestellpositionen
  _positionen := xpath( '/IDOC/E1EDP01', _xml );                                                                                        -- Knoten der Bestellpositionen
  FOREACH _position IN ARRAY _positionen LOOP
    _ldsdok_pos.ld_pos          := z_50_customer.xml__node__value__read( '/E1EDP01/POSEX', _position );                                 -- Bestellposition

    _ldsdok_pos.ld_stk          := z_50_customer.xml__node__value__read( '/E1EDP01/MENGE', _position );                                 -- Stückzahl
    _ldsdok_pos.ld_me           := z_50_customer.xml__node__value__read( '/E1EDP01/MENEE', _position );                                 -- Mengeneinheit
    _ldsdok_pos.ld_netto_grund  := z_50_customer.xml__node__value__read( '/E1EDP01/E1EDP05[KSCHL][KOEIN]/BETRG', _position, false );    -- Nettopreis der Position vor Abzuschlägen
    _ldsdok_pos.ld_netto_grund  := coalesce( _ldsdok_pos.ld_netto_grund, 0 );
    _ldsdok_pos.ld_netto        := z_50_customer.xml__node__value__read( '/E1EDP01/NETWR', _position );                                 -- Nettopreis der Position
    _ldsdok_pos.ld_aknr         := z_50_customer.xml__node__value__read( '/E1EDP01/E1EDP19/IDTNR', _position );                         -- PRODAT-Artikelnummer
    _ldsdok_pos.ld_akbez        := z_50_customer.xml__node__value__read( '/E1EDP01/E1EDP19/KTEXT', _position );                         -- Artikelbezeichnung
    _ldsdok_pos.ld_ekref        := z_50_customer.xml__node__value__read( '/E1EDP01/E1EDP02[QUALF=''045'']/BELNR', _position, false );   -- Kundenreferenz

    _abzus  := xpath( '/E1EDP01/E1EDP05[not(KOEIN)]', _position );                                                                      -- Knoten der Abzuschläge ohne Grundpreis
    _ldsdok_pos.abzus := null::z_50_customer.loll__cera__ldsdok__pos__abzu[];
    IF _abzus IS NOT null THEN
      FOREACH _abzu IN ARRAY _abzus LOOP
        _ldsdok_pos_abzu.ldaz_txt := z_50_customer.xml__node__value__read( '/E1EDP05/KOTXT', _abzu );                                                   -- Bezeichung des Abzuschlags
        _sign := z_50_customer.loll__sign__get( '/E1EDP05/ALCKZ', _abzu );                                                                              -- Vorzeichen des Abzuschlags
        _ldsdok_pos_abzu.ldaz_betrag := _sign * z_50_customer.xml__node__value__read( '/E1EDP05/BETRG', _abzu )::numeric;                               -- Betrag des Abzuschlags (vorzeichenbehaftet)
        _ldsdok_pos_abzu.ldaz_proz   := _sign * translate( z_50_customer.xml__node__value__read( '/E1EDP05/KPERC', _abzu, false ), '-', '' )::numeric;  -- prozentualer Abzuschlag (vorzeichenbehaftet)

        _ldsdok_pos.abzus = array_append( _ldsdok_pos.abzus, _ldsdok_pos_abzu );
      END LOOP;
    END IF;

    _ldsdok.ld_poss := array_append( _ldsdok.ld_poss, _ldsdok_pos );

  END LOOP;

  RETURN _ldsdok loll__cera__ldsdok;

END $$ LANGUAGE plpgsql IMMUTABLE STRICT;
--

-- Sucht eine LOLL-Adresse aus einem Ceratizit-Bestellbestätigungs-XML
-- Der Paramater _tag bezeichnet das XML-Attribut, welches die Adressarten unterscheidet.
CREATE OR REPLACE FUNCTION z_50_customer.loll__adk__find( _adk z_50_customer.loll__cera__adk ) RETURNS varchar AS $$
DECLARE
  _ad_krz varchar;
BEGIN

  -- Findet LOLL-Adresse in PRODAT anhand übergebener Adressdaten.
  -- Bei mehreren infrage kommenden Adressen wird die neueste genommen.
  _ad_krz :=
    ad_krz FROM adressen_view WHERE
        ad_str = _adk.ad_str
    AND ad_plz = _adk.ad_plz
    AND _adk.ad_fa1 ILIKE 'LOLL%'
    AND ad_fa1 ILIKE 'LOLL%'
    ORDER BY modified_date DESC
    LIMIT 1;

  IF _ad_krz IS null THEN
    RAISE EXCEPTION 'Adresse nicht gefunden: %, %, %, %', _adk.ad_fa1, _adk.ad_str, _adk.ad_plz, _adk.ad_ort;
  END IF;

  RETURN _ad_krz;

END $$ LANGUAGE plpgsql STRICT STABLE;
--

-- Code: Funktion zum Anlegen der Bestellpositionen aus einer Ceratizit-XML-Rückmeldung
CREATE OR REPLACE FUNCTION z_50_customer.loll__cera__ldsdok__create( _xml_content xml ) RETURNS varchar AS $$
DECLARE
  _ld_auftg varchar;
  _cent_toleranz numeric;
  _ldsdok z_50_customer.loll__cera__ldsdok;
  _ld_kn varchar;
  _ld_krzl varchar;
  _ld_krzf varchar;
  _ad2 adk2;
  _s varchar;
  _pos z_50_customer.loll__cera__ldsdok__pos;
  _aknr varchar;
  _mce integer;
  _ld_id integer;
  _ld ldsdok;
  _abzu z_50_customer.loll__cera__ldsdok__pos__abzu;
  _abz_id integer;
  _steu numeric;
  _netto numeric;
  _brutto numeric;
  _cera_brutto numeric;
  _ld_konto integer;
  _rabatt numeric;
  _lieferdatum date;
BEGIN

  _ld_auftg := null;

  -- Bei Prüfungen ist eine Differenz von zwei Cent in Ordnung.
  _cent_toleranz := 0.02;

  -- XML in SQL-Struktur einlesen
  _ldsdok := z_50_customer.loll__cera__ldsdok__xml__read( _xml_content );

  -- Adresse Ceratizit auslesen
  _ld_kn := 'WNT';  -- Adresse Ceratizit
  IF NOT EXISTS( SELECT 1 FROM adk WHERE ad_krz = _ld_kn ) THEN
    RAISE EXCEPTION 'Lieferantenadresse nicht gefunden: %', _ld_kn;
  END IF;

  -- Kreditorendaten auslesen
  SELECT * INTO _ad2 FROM adk2 WHERE a2_krz = _ld_kn;
  IF _ad2 IS null THEN
    RAISE EXCEPTION 'Kreditorendaten nicht gefunden %', _ld_kn;
  END IF;

  -- Lieferantenadresse prüfen
  _ld_krzl := z_50_customer.loll__adk__find( _ldsdok.ld_krzl );
  IF _ld_krzl IS null THEN
    RAISE EXCEPTION 'Lieferadresse nicht gefunden: %, %, %, %', _ldsdok.ld_krzl.ad_fa1, _ldsdok.ld_krzl.ad_str, _ldsdok.ld_krzl.ad_ort, _ldsdok.ld_krzl.ad_plz;
  END IF;

  -- Rechnungsadresse prüfen
  _ld_krzf := z_50_customer.loll__adk__find( _ldsdok.ld_krzf );
  IF _ld_krzf IS null THEN
    RAISE EXCEPTION 'Rechnungsadresse nicht gefunden: %, %, %, %', _ldsdok.ld_krzf.ad_fa1, _ldsdok.ld_krzf.ad_str, _ldsdok.ld_krzf.ad_ort, _ldsdok.ld_krzf.ad_plz;
  END IF;

  -- Zahlungsbedingungen prüfen
  IF _ldsdok.a2_sks <> _ad2.a2_sks OR _ldsdok.a2_skv <> _ad2.a2_skv OR _ldsdok.a2_zak <> _ad2.a2_zak THEN
    _s := 'Übertragene Zahlungsbedingungen weichen von Stammdaten ab: ';
    IF _ldsdok.a2_zak <> _ad2.a2_zak THEN
      _s:= _s || 'Zahlungsziel ' || coalesce( _ldsdok.a2_zak, 0 ) || 'd -- ' || coalesce( _ad2.a2_zak, 0 ) || 'd ';
    END IF;
    IF _ldsdok.a2_skv <> _ad2.a2_skv THEN
      _s:= _s || 'Skontoverfall ' || coalesce( _ldsdok.a2_skv, 0 ) || 'd -- ' || coalesce( _ad2.a2_skv, 0 ) || 'd ';
    END IF;
    IF _ldsdok.a2_sks <> _ad2.a2_sks THEN
      _s:= _s || 'Skontosatz ' || coalesce( _ldsdok.a2_sks, 0 ) || '% -- ' || coalesce( _ad2.a2_sks, 0 ) || '% ';
    END IF;
    RAISE EXCEPTION '%', _s;
  END IF;

  -- Prüfen, ob die angekündigte mit der tatsächlichen Zahl der übermittelten Bestellpositionen übereinstimmt
  IF _ldsdok.ld_pos_anz <> array_length( _ldsdok.ld_poss, 1 ) THEN
    RAISE EXCEPTION 'Abweichende Zahl von Bestellpositionen, % -- %', _ldsdok.ld_pos_anz, coalesce( array_length( _ldsdok.ld_poss, 1 ), 0 );
  END IF;

  -- die einzelnen Bestellpositionen anlegen
  IF _ldsdok.ld_poss IS NOT null THEN

    -- Erlöskonto "Werkzeuge"
    _ld_konto := 4985;

    FOREACH _pos IN ARRAY _ldsdok.ld_poss LOOP
      -- Position bereits in PRODAT vorahangen? Dann übergehe sie.
      IF EXISTS( SELECT 1 FROM ldsdok WHERE ld_kn = _ld_kn AND ld_post6 = _ldsdok.ld_post6 AND ld_pos = _pos.ld_pos ) THEN
        CONTINUE;
      END IF;

      -- Auftragsnummer erst nach Duplikatsprüfung ermitteln, zur Vermeidung von Leerstellen
      IF _ld_auftg IS null THEN
        _ld_auftg := nummer FROM teinkauf.generate_bestellnummer_ext_by_agnr( 'E' );
      END IF;

      -- Artikel in PRODAT suchen
      _aknr := ak_nr FROM art WHERE ak_nr = _pos.ld_aknr;
      IF _aknr IS null THEN
        -- Wenn Artikel nicht gefunden, dann neu anlegen mit AC 5000 (Werkzeuge).
        INSERT INTO art
          ( ak_nr, ak_ac, ak_bez, ak_standard_mgc )
        VALUES
          ( _pos.ld_aknr, 5000, _pos.ld_akbez, 1 );

        _aknr := _pos.ld_aknr;
      END IF;

      -- Mengencode für "Stück" auslesen
      _mce := m_id FROM artmgc WHERE m_ak_nr = _aknr AND m_mgcode = 1;
      IF _mce IS null THEN
        RAISE EXCEPTION 'Kein Mengencode für "Stück": Artikelnummer % Position %',  _pos.ld_aknr, _pos.ld_pos;
      END IF;

      -- Lieferfrist mindestens sieben Tage
      _lieferdatum := timediff_adddays( _ldsdok.ld_datum, 7, false, true );
      IF _ldsdok.ld_term < _lieferdatum THEN
        _ldsdok.ld_term := _lieferdatum;
      END IF;

      --Rabatt
      _rabatt := 0;
      IF _pos.abzus IS NOT null THEN
         FOREACH _abzu IN ARRAY _pos.abzus LOOP
           -- Rabatte werden als negativer Wert übermittelt
           _rabatt := _rabatt - coalesce( _abzu.ldaz_proz, 0 );
         END LOOP;
      END IF;  

      -- Einfügen der Bestellposition
      _ld_id := null;
      INSERT INTO ldsdok
        ( ld_code, ld_auftg, ld_pos, ld_kn, ld_krzl, ld_krzf, ld_aknr, ld_stk, ld_abnr, ld_abdat,
          ld_term, ld_terml, ld_datum, ld_ep, ld_steuproz, ld_mce, ld_waer, ld_post6, ld_post7, ld_konto, 
          ld_arab,
          ld_ekref, insert_date, insert_by, modified_date, modified_by
        )
      VALUES
        ( 'E', _ld_auftg, _pos.ld_pos, _ld_kn, _ld_krzl, _ld_krzf, _aknr, _pos.ld_stk, _ldsdok.ld_post6, _ldsdok.ld_datum, 
          _ldsdok.ld_term, _ldsdok.ld_term, _ldsdok.ld_datum, _pos.ld_netto_grund / _pos.ld_stk, _ldsdok.ld_steuproz, _mce, _ldsdok.ld_waer, _ldsdok.ld_post6, _ldsdok.ld_post7, _ld_konto, 
          _rabatt,
          _pos.ld_ekref, now(), 'CIMPCS', now(), 'CIMPCS'
        )
      RETURNING ld_id INTO _ld_id;
      -- Bestellposition auslesen
      SELECT * INTO _ld FROM ldsdok WHERE ld_id = _ld_id;

      -- Test, ob übertragene und errechnete Nettobeträge der Position übereinstimmen
      IF @( _ld.ld_netto - _pos.ld_netto ) > _cent_toleranz THEN
        RAISE EXCEPTION 'Übertragener und gespeicherter Nettobetrag weichen ab bei Position %: % -- %', _pos.ld_pos, _pos.ld_netto, _ld.ld_netto;
      END IF;

    END LOOP;

  END IF;

  -- Vergleich der Mehrwertsteuerbeträge
  _steu := sum( ld_brutto - ld_netto ) FROM ldsdok WHERE ld_auftg = _ld_auftg;
  IF @( _steu - _ldsdok.ld_steu ) > _cent_toleranz THEN
    RAISE EXCEPTION 'Übertragener und gespeicherter MwSt-betrag weichen ab : % -- %', _ldsdok.ld_steu, _steu;
  END IF;

  -- Vergleich der Nettobeträge
  _netto := sum( ld_netto ) FROM ldsdok WHERE ld_auftg = _ld_auftg;
  IF @( _netto - _ldsdok.ld_netto ) > _cent_toleranz THEN
    RAISE EXCEPTION 'Übertragener und gespeicherter Nettobetrag weichen ab : % -- %',_ldsdok.ld_netto, _netto;
  END IF;

  -- Vergleich der Bruttobeträge
  _brutto := sum( ld_brutto ) FROM ldsdok WHERE ld_auftg = _ld_auftg;
  _cera_brutto := _ldsdok.ld_steu + _ldsdok.ld_netto;
  IF @( _brutto - _cera_brutto ) > _cent_toleranz THEN
    RAISE EXCEPTION 'Übertragener und gespeicherter Bruttobetrag weichen ab : % -- %', _cera_brutto, _brutto;
  END IF;

  -- Wurde nichts angelegt, weil alle Bestellpositionen bereits in PRODAT, dann Nummer dieser Bestellung zurückgeben
  IF _ld_auftg IS null THEN
    _ld_auftg := ld_auftg FROM ldsdok WHERE ld_kn = _ld_kn AND ld_post6 = _ldsdok.ld_post6 LIMIT 1;
  END IF;

  RAISE NOTICE 'Import Ceratizit-Beleg % abgeschlossen', _ldsdok.ld_post6;
  RETURN _ld_auftg;

END $$ LANGUAGE plpgsql STRICT;
--

-- Funktion zum Auslesen der EDI-Datensätze aus der Tabelle "edi_incomming_xml"
CREATE OR REPLACE FUNCTION z_50_customer.loll__cera__order__confirmation__parse( _edi_origin varchar ) RETURNS varchar AS $$
DECLARE
  _edi_id varchar;
  _xml_content xml;
  _ld_auftg varchar;
  _err_msg text;
BEGIN

  _edi_id := edi_id FROM edi_incomming_xml WHERE edi_origin = _edi_origin AND edi_type = 'Auftragsbestätigung Ceratizit';
  _xml_content := edi_content FROM edi_incomming_xml WHERE edi_id = _edi_id;

  _err_msg := null; _ld_auftg := null;
  BEGIN
    _ld_auftg := z_50_customer.loll__cera__ldsdok__create( _xml_content );
  EXCEPTION WHEN others THEN
    _err_msg := sqlerrm;
  END;

  UPDATE edi_incomming_xml SET edi_processing = now(), edi_error_message = _err_msg, edi_success = _ld_auftg WHERE edi_id = _edi_id;
  RETURN _ld_auftg;

END $$ language plpgsql STRICT;
--